/* Last Change: 20010429 - doe */
/*
 *  $Id: P54-06,v 1.16 1998/12/10 00:01:28 route Exp $
 *  Trusted path ACL implementation for OpenBSD 2.4
 *
 *  Copyright (c) 1998 route|daemon9 and Mike D. Schiffman
 *  All rights reserved.
 *  Originally published in Phrack Magazine (http://www.phrack.com).
 *
 *  Thanks to nirva for helping me choose an ADT.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      Trusted path ACL implementation for OpenBSD 2.4
 *
 *      For the full write-up please see Phrack Magazine, issue 54, article 6
 *      http://www.phrack.com
 *
 *      Overview:
 *
 *  A trusted path/ACL execution implementation for OpenBSD.  We consider
 *  a path to be trusted if the parent directory is owned by root and is not
 *  group or world writable.  We consider a user to be trusted if she is on
 *  the kernels trust list.
 *      Implementation details:
 *
 *  Inside the kern_exec function, we first check the path for trust, if that
 *  fails, we then check the user's credentials to see if she is  able to run
 *  binaries in an untrusted path.  Untrusted users are not allowed to execute
 *  programs from untrusted pathes.
 *
 *  The decision was the made to use a static array to hold the trusted IDs
 *  for both convienience and runtime efficiency.  We keep the list ordered
 *  after all insertions and deletions, and therefore, we can search the list
 *  (where speed is critical) in a worst case of O(lg N).  Compare that with a
 *  sequential search in an ordered list which has a worst case of O(N).
 *
 *  The speed in which user ID verification is done is absolutely essential,
 *  as this check will be done for every call to exec that does not originate
 *  from a trusted path.  This has the potential to be a huge bottle neck.
 *  This was taken into consideration and the bulk of processing overhead is
 *  offloaded to list initialization and modification.
 */

#ifndef __KERN_TPE_H
#define __KERN_TPE_H

#ifdef _KERNEL
#include <sys/types.h>
#include <sys/cdefs.h>
#include <sys/systm.h>
#include <sys/param.h>
#include <sys/ucred.h>
#include <sys/proc.h>
#endif

/*
 *  syscall stuff
 */
#define TPE_ADD         0       /* add an entry */
#define TPE_REMOVE      1       /* delete an entry */
#define TPE_SHOW        2       /* show the list */
#define TPE_VERIFY	4	/* verify a given uid is in the list */

#define TPE_ACL_SIZE    80      /* Shouldn't need to be larger */
#define TPE_INITIALIZER -1      /* A UID that isn't used */

#define ACK             1       /* positive acknowledgement */
#define NACK            -1      /* negative acknowledgement */

#define LESS(X, Y)      (X < Y)
#define SWAP(X, Y)      (X ^= Y, Y ^= X, X ^= Y)
#define COMPSWAP(X, Y)  if (LESS(Y, X)) SWAP(X, Y)

/*
 *  Verify the path.  This macro is passed a filled in attr struct via
 *  VOP_GETATTR.
 */
#define TRUSTED_PATH(AT) \
(!(AT.va_mode & (S_IWGRP | S_IWOTH)) && (AT.va_uid == 0))

/*
 *  Verify the user.  This macro is passed the user's ID from the u_cred
 *  struct.
 */
#define TRUSTED_USER(UID) (tpe_verify(UID) == ACK)

uid_t tpe_acl[TPE_ACL_SIZE];    /* trusted user list */
int tpe_acl_candidates;         /* number of users on the list */
int tpe_ld_check;               /* check ld.so env */

/*
 *  Initialize the array with default values (TPE_INITIALIZER).
 */
void
tpe_init __P((
    void
    ));


/*
 *  Dump the list.
 */
void
tpe_show __P((
    void
    ));


/*
 *  Attempt to add a candidate to the list.  Only fails if the list is full.
 */
int
tpe_add __P((
    uid_t       /* candidate user for addition */
    ));


/*
 *  Attempt to remove a candidate from the list.  Only fails if the entry is
 *  not there.
 */
int
tpe_remove __P((
    uid_t       /* candidate user for deletion */
    ));


/*
 *  Verify a candidate user.
 */
int
tpe_verify __P((
    uid_t       /* candidate user for verification */
    ));


/*
 *  Insertion sort the list.
 */
void
tpe_sort __P((
    int,        /* list low element */
    int         /* list high high element */
    ));


/*
 *  Locate a uid in the list, standard recursive binary search, running in
 *  worst case of lg N.
 */
int
tpe_search __P((
    uid_t,      /* candidate user to search for */
    int,        /* list low element */
    int         /* list high high element */
    ));

#endif  /* __KERN_TPE_H */
