/*
 *      faxrx.c -- HF Fax (WEFAX) demodulator software
 *
 *	Copyright (C) 1996  
 *          Thomas Sailer (sailer@ife.ee.ethz.ch, hb9jnx@hb9w.che.eu)
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Current state of the affairs
 *   The FM demodulator should work pretty well. The other things are
 *   more or less dirty hacks. My documentation about the format of
 *   fax transmissions is very sparse, so someone with a better knowledge
 *   of HF fax should probably improve it.
 *
 * How does it work
 *   The FM demodulator is fed with a 16 bit linear 8kSamples/s monaural
 *   sample stream. Sox can be used to convert various audio file formats
 *   to this format. Alternatively, the samples may also come from the
 *   sound driver.
 *   The center frequency of the FM demodulator is 1500 Hz, and it handles
 *   deviations of +-400 Hz. The bandwith at the input is approx. 2.5 kHz.
 *   The output is fed to the synchronisation and sampling logic.
 *   The goal of the synchronisation logic is to find the beginning of the
 *   picture, adjust black and white levels, find out the drum rotation speed,
 *   and afterwards to sample the FM demodulator output at the right times
 *   and save the samples to disk. 
 *   The synchronisation is currently done in three steps.
 *   The first searches for a continuous tone. This should prevent the 
 *   demodulator from detecting sync during just noise.
 *   The second step searches for the repeated black and white transitions
 *   that are usually (?) found at the beginning of fax transmissions.
 *   These pattern is also used to set the black and white levels.
 *   The first few lines of a transmission should be black with a white bar
 *   at the beginning/end of the line. The third stage searches for these
 *   lines and uses them to determine the beginning of a scan line and the
 *   drum rotation speed.
 *   The picture is currently saved to disk in the binary PGM (portable
 *   gray map) format. The receiving ends after a preset number of lines.
 *
 * To do:
 *   Someone should write a nice X user interface. This will probably have
 *   some impact on the sample input and the picture output.
 *   The synchronisation logic should be improved. It should be tuned for
 *   automatic reception, but still let the user influence its operation.
 *   Also the drum rotation speed isn't measured exactly enough, so the
 *   picture currently still has some skew.
 *   
 *    
 */

/* ---------------------------------------------------------------------- */

#include "soundif.h"
#include "xgraphif.h"
#include "fir_filters.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <math.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/time.h>
#include <termios.h>

/* ---------------------------------------------------------------------- */

static const char *allowed_types[] = {
	"raw", "aiff", "au", "hcom", "sf", "voc", "cdr", "dat", 
	"smp", "wav", "maud", "vwe", NULL
};

/* ---------------------------------------------------------------------- */

static char *input_type = "hw";

/* ---------------------------------------------------------------------- */

#define INPUT_LPF_LEN 32
#define DIFFERENTIATOR_LEN 9
#define OUTPUT_LPF_LEN 32

/* ---------------------------------------------------------------------- */

static const float lpf_coeff_symm[(INPUT_LPF_LEN+1)/2] = {
              -0.000193,        0.001542,        0.002594,        0.000807,
              -0.004534,       -0.008656,       -0.003458,        0.011956,
               0.023448,        0.011107,       -0.026607,       -0.057711,
              -0.033850,        0.066649,        0.207570,        0.310332
};

static const float diff_coeff_symm[(DIFFERENTIATOR_LEN+1)/2] = {
              -0.006366,        0.022784,       -0.085944,        0.275424,
               0.000000
};
/*
 * note: the differentiator linearity is quite good up to 2kHz,
 * but it has some gain of approx 1.6
 * NOTE: the differentiator length must be odd!
 */

static const float olpf_coeff_symm[(OUTPUT_LPF_LEN+1)/2] = {
               0.001464,        0.001749,        0.000433,       -0.002925,
              -0.006099,       -0.004083,        0.005784,        0.017397,
               0.016905,       -0.005020,       -0.038292,       -0.051780,
              -0.012648,        0.084828,        0.205172,        0.288337
};

/* ---------------------------------------------------------------------- */

#define BUFFER_LEN 512 /* samples */
#define SINE_TAB_LEN 1024 /* must be a power of 2! */
#define SAMPLING_RATE 8000 /* DON'T touch */
#define CENTER_FREQ 1500

#define TRACE_INPUT 1
#define TRACE_DEMODOUT 2
#define TRACE_LEN 1024

#define SYNC_HUNT_IDLE 0
#define SYNC_HUNT_BW   1
#define SYNC_HUNT_LINE 2
#define SYNC_HUNT_RX   3

struct fax_demod_state {
	int mixer_ph;
	int mixer_inc;
	float i_diff[DIFFERENTIATOR_LEN];
	float q_diff[DIFFERENTIATOR_LEN];
	float i_lpf[INPUT_LPF_LEN];
	float q_lpf[INPUT_LPF_LEN];
	float o_lpf[OUTPUT_LPF_LEN];
	float sintab[SINE_TAB_LEN];
	int sample_phase;
	int trace_flags;
	int trace_cnt;

	int sync_mode;
	float sync_minval;
	float sync_maxval;
	float sync_multval;
	int sync_width;
	int sync_frac_width;
	int sync_width_phase;
	int sync_width_phaseinc;
	int sync_height;
	int height;
	int width;
	int frac_width;
	int sync_cnt1;
	int sync_cnt2;
	int sync_cnt3;
	int syncbuf[32];

	FILE *faxfile;
};

struct fax_demod_state fax_state;

/* ---------------------------------------------------------------------- */

static void fax_sync_init(void)
{
	fax_state.sync_mode = SYNC_HUNT_IDLE;
	fax_state.sync_cnt1 = fax_state.sync_cnt2 = 0;
}

/* ---------------------------------------------------------------------- */

static void fax_file_close(void)
{
	if (!fax_state.faxfile)
		return;
	if (fax_state.sync_width || fax_state.sync_height)
		for(; fax_state.sync_height < fax_state.height;
		    fax_state.sync_height++) {
			for(; fax_state.sync_width < fax_state.width;
			    fax_state.sync_width++) 
				putc(128, fax_state.faxfile);
			fax_state.sync_width = 0;
		}
	fclose(fax_state.faxfile);
}

/* ---------------------------------------------------------------------- */

static void fax_demod_init(void)
{
	int i;

	for(i = 0; i < SINE_TAB_LEN; i++)
		fax_state.sintab[i] = sin(((float)i) * 2.0 * M_PI
					  / SINE_TAB_LEN);
	fax_state.mixer_ph = 0;
	fax_state.mixer_inc = SINE_TAB_LEN * CENTER_FREQ / SAMPLING_RATE;
	/*
	 * initialize FIR filter memory; not strictly necessary,
	 * but otherwise the filters output NaN the first few cycles
	 */
	for(i = 0; i < DIFFERENTIATOR_LEN; i++)
		fax_state.i_diff[i] = fax_state.q_diff[i] = 0;
	for(i = 0; i < INPUT_LPF_LEN; i++)
		fax_state.i_lpf[i] = fax_state.q_lpf[i] = 0;
	for(i = 0; i < OUTPUT_LPF_LEN; i++)
		fax_state.o_lpf[i] = 0;
	fax_state.trace_cnt = fax_state.trace_flags = 0;
	fax_state.sample_phase = 0;
	fax_state.faxfile = NULL;
	fax_state.height = 2300;
	fax_sync_init();
}

/* ---------------------------------------------------------------------- */

static __inline__ unsigned int hamming_weight(unsigned int w)
{
	unsigned int res = (w & 0x55555555) + ((w >> 1) & 0x55555555);
	res = (res & 0x33333333) + ((res >> 2) & 0x33333333);
	res = (res & 0x0F0F0F0F) + ((res >> 4) & 0x0F0F0F0F);
	res = (res & 0x00FF00FF) + ((res >> 8) & 0x00FF00FF);
	return (res & 0x0000FFFF) + ((res >> 16) & 0x0000FFFF);
}

/* ---------------------------------------------------------------------- */

static __inline__ void fax_sync_hunt_idle(float val)
{
	fax_state.sync_cnt1++;
	fax_state.sync_cnt2 = (fax_state.sync_cnt2 << 1) | (val > 0);
	if ((fax_state.sync_cnt2 ^ (fax_state.sync_cnt2 >> 1)) & 1)
		fax_state.sync_cnt1 = 0;
	if (fax_state.sync_cnt1 < 128)
		return;
	fax_state.sync_mode = SYNC_HUNT_BW;
	fax_state.sync_minval = fax_state.sync_maxval = 0;
	fax_state.sync_multval = 0.02;
	fax_state.sync_cnt1 = fax_state.sync_cnt2 = 0;
	memset(fax_state.syncbuf, 0, sizeof(fax_state.syncbuf));
	fprintf(stdout, "start tone detected (?)\n");
}

/* ---------------------------------------------------------------------- */

static __inline__ void fax_sync_hunt_bw(float val)
{
	int i, j;
	int *ip;

	if(val >= fax_state.sync_maxval)
		fax_state.sync_maxval = val;
	else
		fax_state.sync_maxval *= 0.999;
	if(val <= fax_state.sync_minval)
		fax_state.sync_minval = val;
	else
		fax_state.sync_minval *= 0.999;
	if (val > fax_state.sync_multval && 
	    fax_state.sync_multval >= 0) {
		fax_state.sync_multval = -0.02;
		fax_state.sync_cnt2++;
	} else if (val < fax_state.sync_multval && 
		   fax_state.sync_multval <= 0) {
		fax_state.sync_multval = 0.02;
		fax_state.sync_cnt2++;
	}
	if ((++fax_state.sync_cnt1) < 128) 
		return;
	ip = fax_state.syncbuf+14;
	for(i = j = 0; i <= 14; i++, ip--)
		j += (ip[1] = ip[0]);
	j += (fax_state.syncbuf[0] = fax_state.sync_cnt2);
#if 0
	fprintf(stdout, "number of transitions now: %d sum: %d\n", 
		fax_state.sync_cnt2, j);
#endif
	if (j <= 1024 && j >= 200 && fax_state.sync_cnt2 <= 64 &&
	    fax_state.sync_cnt2 >= 12) {
		fax_state.sync_multval = 255.0 / 
			(fax_state.sync_maxval - fax_state.sync_minval);
		fax_state.sync_mode = SYNC_HUNT_LINE;
		fax_state.sync_cnt1 = 0;
		fax_state.sync_cnt2 = 0x55555555;
		fax_state.sync_cnt3 = 0;
		fprintf(stdout, "Black&White pattern found, "
			"now trying to acquire line sync\n");
		return;
	}
	fax_state.sync_cnt2 = fax_state.sync_cnt1 = 0;
}

/* ---------------------------------------------------------------------- */

static __inline__ void fax_sync_hunt_line(float val)
{
	struct oneline {
		int len;
		int start;
		int inv;
	};
	struct oneline lines[32];
	int i, j, k, l;

	fax_state.sync_cnt3++;
	fax_state.sync_cnt2 = (fax_state.sync_cnt2 << 1) | (val > 0);
	/*
	 * do not count lines which seem to be much faster than 240rpm
	 */
	if (fax_state.sync_cnt1 >= 2 && 
	    (fax_state.sync_cnt3 - fax_state.syncbuf[fax_state.sync_cnt1-2])
	    < 400)
		return;
	/*
	 * find transitions
	 */
	if (fax_state.sync_cnt1 & 1) {
		/* expect a 1-0 transition */
		if (((fax_state.sync_cnt2 & 0x00018000) == 0x00010000) && 
		    (hamming_weight(fax_state.sync_cnt2 & 0xffff0000) >= 12) &&
		    (hamming_weight(fax_state.sync_cnt2 & 0x0000ffff) <= 4)) 
			fax_state.syncbuf[fax_state.sync_cnt1++] =
				fax_state.sync_cnt3;
	} else {
		/* expect a 0-1 transition */
		if (((fax_state.sync_cnt2 & 0x00018000) == 0x00008000) && 
		    (hamming_weight(fax_state.sync_cnt2 & 0xffff0000) <= 4) &&
		    (hamming_weight(fax_state.sync_cnt2 & 0x0000ffff) >= 12))
			fax_state.syncbuf[fax_state.sync_cnt1++] =
				fax_state.sync_cnt3;
	}
	if (fax_state.sync_cnt1 < 32)
		return;
	if (fax_state.sync_cnt3 > 100000) {
		fax_state.sync_mode = SYNC_HUNT_IDLE;
		return;		
	}
	/*
	 * line sync found, check if we must invert
	 */
	/*
	 * first sort line lengths found
	 */
	for(i = 0; i < 30; i++) {
		j = fax_state.syncbuf[i+2] - fax_state.syncbuf[i];
		for(k = 0; (k < i) && (j > lines[k].len); k++);
		if (k >= i)
			k = i;
		else
			for(l = i; l > k; l--)
				lines[l] = lines[l-1];
		lines[k].len = j;
		l = fax_state.syncbuf[i+1] > ((fax_state.syncbuf[i+2] + 
					       fax_state.syncbuf[i])/2);
		lines[k].inv = l == (i & 1);
		lines[k].start = l ? ((fax_state.syncbuf[i+1]+
				       fax_state.syncbuf[i+2])/2) :
					       ((fax_state.syncbuf[i]+
						 fax_state.syncbuf[i+1])/2);
	}
	for(i = 0; i < 30; i++)
		fprintf(stdout, "line %d width %d start %d%s\n", i, 
			lines[i].len, lines[i].start, 
			lines[i].inv ? " inv" : "");
	/*
	 * take the median and use only lines which do not differ significantly
	 * from it
	 */
	for(i = 0; (i < 14) && 
	    (abs(lines[i].len - lines[14].len) > (lines[14].len / 16)); i++);
	for(j = 29; (j > 14) && 
	    (abs(lines[j].len - lines[14].len) > (lines[14].len / 16)); j--);
	for(k = i, l = 0; k <= j; k++)
		l += lines[k].len;
	l = (l * 256 + (j-i+1)/2) / (j-i+1);
	fprintf(stdout, "width %d from %d lines\n", (l+128)/256, (j-i+1));
	fax_state.width = l / 256;
	fax_state.frac_width = l;
	fax_state.sync_cnt3 = lines[14].start - fax_state.sync_cnt3;
	if (lines[14].inv) {
		fprintf(stdout, "inverting picture\n");
	} else {
		fax_state.sync_minval = fax_state.sync_maxval;
		fax_state.sync_multval = -fax_state.sync_multval;
	}
	while (fax_state.sync_cnt3 < 0)
		fax_state.sync_cnt3 += fax_state.width;
	fax_state.sync_width_phaseinc = 256;
	while (2*fax_state.width > 3*fax_state.height) {
		fax_state.width /= 2;
		fax_state.sync_width_phaseinc /= 2;
	}
	fax_state.sync_width = fax_state.sync_frac_width = 0;
	fax_state.sync_height = fax_state.sync_width_phase = 0;
	fax_state.sync_mode = SYNC_HUNT_RX;
	if (!(fax_state.faxfile = fopen("testpic.pgm", "w"))) {
		fprintf(stdout, "error, cannot open fax picture file \"%s\"\n",
			"testpic.pgm");
		fax_state.sync_mode = SYNC_HUNT_BW;
		return;
	}
	fprintf(fax_state.faxfile, "P5 %d %d 255\n", fax_state.width, 
		fax_state.height);
	fprintf(stdout, "line sync found, receiving\n");
	fprintf(stdout, "picture size: width %d (fractional %f) height %d\n",
		fax_state.width, (double)fax_state.frac_width / 256.0, 
		fax_state.height);
}

/* ---------------------------------------------------------------------- */

static __inline__ void fax_sync_rx(float val)
{
	float f;

	if (fax_state.sync_cnt3 > 0) {
		fax_state.sync_cnt3--;
		return;
	}
	fax_state.sync_width_phase += fax_state.sync_width_phaseinc;
	if (fax_state.sync_width_phase >= 256) {
		fax_state.sync_width_phase -= 256;
		if (fax_state.sync_width < fax_state.width) {
			f = (val - fax_state.sync_minval) * 
				fax_state.sync_multval;
			if (f > 255)
				f = 255;
			if (f < 0) 
				f = 0;
			putc((int)f, fax_state.faxfile);
			fax_state.sync_width++;
		}
	}
	fax_state.sync_frac_width += 256;
	if (fax_state.sync_frac_width < fax_state.frac_width)
		return;
	if (fax_state.sync_width < fax_state.width) {
		putc(0, fax_state.faxfile);
		fprintf(stdout, "error: line too short\n");
	}
	fax_state.sync_frac_width -= fax_state.frac_width;
	fax_state.sync_width_phase = fax_state.sync_width = 0;
	if ((++fax_state.sync_height) >= fax_state.height) {
		fclose(fax_state.faxfile);
		fax_state.faxfile = NULL;
		fprintf(stdout, "finished!\n");
		exit(0);
	}
	if (!(fax_state.sync_height % 10))
		fprintf(stdout, "%d lines of %d received\n", 
			fax_state.sync_height, fax_state.height);
}

/* ---------------------------------------------------------------------- */

static __inline__ void fax_sample(float val)
{
	switch (fax_state.sync_mode) {
	default:
		fax_sync_init();
		return;
	case SYNC_HUNT_IDLE:
		fax_sync_hunt_idle(val);
		return;
	case SYNC_HUNT_BW:
		fax_sync_hunt_bw(val);
		return;
	case SYNC_HUNT_LINE:
		fax_sync_hunt_line(val);
		return;
	case SYNC_HUNT_RX:
		fax_sync_rx(val);
		return;
	}
}

/* ---------------------------------------------------------------------- */

static int fax_demod_buffer(int infd)
{
	float ilpf, qlpf, idiff, qdiff, indiff, qndiff;
	float energy, demod_out = 0;
	short inbuf[BUFFER_LEN];
	short *inbuf_ptr;
	float trace_val = 0;
	int i, j;
	int need_outval;
	
	i = read(infd, inbuf_ptr = inbuf, sizeof(inbuf));
	if (i < 0) {
		fprintf(stderr, "input error %i\n", errno);
		return -1;
	}
	if (!i) {
		fax_file_close();
		return 1; /* end of file */
	}
	if (i % sizeof(short)) 
		fprintf(stderr, "warning: read nonintegral number "
			"of samples\n");
	for (j = i / sizeof(short); j > 0; j--) {
		float val = ((float)(*inbuf_ptr++)) / 32768.0;
		if (fax_state.trace_flags == TRACE_INPUT)
			trace_val = val;
		ilpf = float_fir_symm(fax_state.sintab[(fax_state.mixer_ph + 
					      (SINE_TAB_LEN/4)) &
					     (SINE_TAB_LEN-1)] * val,
				      fax_state.i_lpf, lpf_coeff_symm, 
				      INPUT_LPF_LEN);
		qlpf = float_fir_symm(fax_state.sintab[fax_state.mixer_ph]
				      * val, fax_state.q_lpf, lpf_coeff_symm, 
				      INPUT_LPF_LEN);
		fax_state.mixer_ph = (fax_state.mixer_ph + 
				      fax_state.mixer_inc) & (SINE_TAB_LEN-1);
		idiff = float_fir_symm_hilbert(ilpf, fax_state.i_diff, 
					       diff_coeff_symm,
					       DIFFERENTIATOR_LEN);
		qdiff = float_fir_symm_hilbert(qlpf, fax_state.q_diff, 
					       diff_coeff_symm,
					       DIFFERENTIATOR_LEN);
		indiff = fax_state.i_diff[DIFFERENTIATOR_LEN/2];
		qndiff = fax_state.q_diff[DIFFERENTIATOR_LEN/2];
		energy = indiff * indiff + qndiff * qndiff;
		if (energy == 0)
			demod_out = 0;
		else 
			demod_out = (qndiff * idiff - indiff * qdiff)
				/ energy;
		/*
		 * determine if we need to calculate the filter output
		 * or if it is sufficient to update the filter state
		 */
		need_outval = !fax_state.sample_phase;
		if (fax_state.trace_flags == TRACE_DEMODOUT)
			need_outval = 1;
		/*
		 * update the filter
		 */
		if (need_outval)
			demod_out = float_fir_symm(demod_out, 
						   fax_state.o_lpf, 
						   olpf_coeff_symm, 
						   OUTPUT_LPF_LEN);
		else {
			memmove(fax_state.o_lpf+1, fax_state.o_lpf, 
				sizeof(fax_state.o_lpf) - sizeof(float));
			fax_state.o_lpf[0] = demod_out;
		}
		/* 
		 * the filtered output of the demodulator
		 * the output range is approx -0.1 .. +0.1
		 * for deviations from -400 Hz .. +400 Hz
		 */
		if (fax_state.trace_flags == TRACE_DEMODOUT)
			trace_val = demod_out;
		/*
		 * output values if necessary
		 */
		if (!fax_state.sample_phase) 
			fax_sample(demod_out);
		fax_state.sample_phase = (fax_state.sample_phase+1) & 1;
		/*
		 * update traced values
		 */
		if (fax_state.trace_flags) {
			xgraph_newdata(XGM_DEFAULT, 
				       ((float)(fax_state.trace_cnt++)) /
				       SAMPLING_RATE, trace_val);
			if (fax_state.trace_cnt >= TRACE_LEN) {
				xgraph_close();
				fax_state.trace_cnt = 
					fax_state.trace_flags = 0;
			}
		}
	}
	fprintf(stdout, "val: %10.5f\r", demod_out);
	fflush(stdout);
	return 0;
}

/* ---------------------------------------------------------------------- */

static void do_file(char *fname, char interactive)
{
	struct stat statbuf;
	int pipedes[2];
	int pid = 0, soxstat;
	int infd;
	fd_set rmask;
	char kbdbuf[16];
	int i;
	char *cp;
	struct termios tmios;

	/*
	 * if the input type is hw, then we use the available sound hardware
	 */
	if (!strcmp(input_type, "hw")) {
		if (soundif_start(8000, O_RDONLY) < 0) {
			fprintf(stderr, "cannot open sound interface \"%s\"\n",
				soundif_name);
			return;
		}
		infd = soundif_fd;
	}
	/*
	 * if the input type is not raw, sox is started to convert the
	 * samples to the requested format
	 */
	else if (!strcmp(input_type, "raw")) {
		if ((infd = open(fname, O_RDONLY)) < 0) {
			fprintf(stderr, "input file \"%s\" does not"
				" exist\n", fname);
			return;
		}
	} else {
		if (stat(fname, &statbuf)) {
			fprintf(stderr, "input file \"%s\" does not"
				" exist\n", fname);
			return;
		}
		if (pipe(pipedes))
			perror("pipe");
		if (!(pid = fork())) {
			/*
			 * child starts here... first set up filedescriptors,
			 * then start sox...
			 */
			close(pipedes[0]); /* close reading pipe end */
			close(1); /* close standard output */
			if (dup2(pipedes[1], 1) < 0) 
				perror("dup2");
			close(pipedes[1]); /* close writing pipe end */
			execlp("sox", "sox", 
			       "-t", input_type, fname,
			       "-t", "raw", "-s", "-w", "-r", "8000", "-",
			       NULL);
			perror("execlp");
			exit(1);
		}
		if (pid < 0)
			perror("fork");
		close(pipedes[1]); /* close writing pipe end */
		infd = pipedes[0];
	}
	/*
	 * start the demodulator
	 */
	fax_demod_init();
	if (interactive) {
		if (tcgetattr(0, &tmios))
			perror("tcgetattr");
		tmios.c_lflag &= (~ICANON);
		tmios.c_cc[VMIN] = 1;
		if (tcsetattr(0, TCSANOW, &tmios))
			perror("tcsetattr");
	}
	for(;;) {
		FD_ZERO(&rmask);
		if (interactive)
			FD_SET(0, &rmask);
		FD_SET(infd, &rmask);
		i = select(infd+1, &rmask, NULL, NULL, NULL);
		if (i < 0) {
			perror("select");
			break;
		}
		if (FD_ISSET(infd, &rmask))
			if (fax_demod_buffer(infd))
				break;
		if (interactive && (FD_ISSET(0, &rmask))) {
			i = read(0, cp = kbdbuf, sizeof(kbdbuf));
			if (i < 0) {
				perror("kbd read");
				break;
			}
			for(; i > 0; i--, cp++) {
				switch(*cp) {
				case 'i':
					if (fax_state.trace_flags)
						break;
					if(xgraph_open(NULL)) {
						perror("xgraph_open");
						break;
					}
					xgraph_settitle("input signal");
					xgraph_newset("in");
					fax_state.trace_cnt = 0;
					fax_state.trace_flags = TRACE_INPUT;
					break;
				case 'd':
					if (fax_state.trace_flags)
						break;
					if(xgraph_open(NULL)) {
						perror("xgraph_open");
						break;
					}
					xgraph_settitle("demodulator output "
							"signal");
					xgraph_newset("demod");
					fax_state.trace_cnt = 0;
					fax_state.trace_flags = TRACE_DEMODOUT;
					break;
				default:
					break;
				}
			}
		}
	}
	/*
	 * end of the demodulator
	 */
	if (!strcmp(input_type, "hw")) 
		soundif_stop();
	else
		close(infd);
	waitpid(pid, &soxstat, 0);
}

/* ---------------------------------------------------------------------- */

static const char usage_str[] = "faxrx\nDemodulates (WE)FAX signals\n"
"(C) 1996 by Thomas Sailer HB9JNX/AE4WA\n"
"  -t <type> : input file type (any other type than raw requires sox)\n";

void main(int argc, char *argv[])
{
	int c;
	int errflg = 0;
	int idx;
	char **itype;

	while ((c = getopt(argc, argv, "t:")) != EOF) {
		switch (c) {
		case '?':
			errflg++;
			break;
		case 't':
			for(itype = (char **)allowed_types; *itype; itype++) 
				if (!strcmp(*itype, optarg)) {
					input_type = *itype;
					goto intypefound;
				}
			fprintf(stderr, "invalid input type \"%s\"\n"
				"allowed types: ", optarg);
			for(itype = (char **)allowed_types; *itype; itype++) 
				fprintf(stderr, "%s ", *itype);
			fprintf(stderr, "\n");
			errflg++;
		intypefound:
			break;
		}
	}
	if (errflg) {
		(void)fprintf(stderr, usage_str);
		exit(2);
	}
	if (!strcmp(input_type, "hw")) {
		if ((argc - optind) >= 1)
			strcpy(soundif_name, argv[optind]);
		do_file(NULL, 1);
		exit(0);
	}
	if ((argc - optind) < 1) {
		(void)fprintf(stderr, "no source files specified\n");
		exit(3);
	}
	for(idx = optind; idx < argc; idx++)
		do_file(argv[idx], 0);
	exit(0);
}

/* ---------------------------------------------------------------------- */

