/*
 *      soundif.c -- access Linux/Solaris sound driver
 *
 *	Copyright (C) 1996  
 *          Thomas Sailer (sailer@ife.ee.ethz.ch, hb9jnx@hb9w.che.eu)
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* ---------------------------------------------------------------------- */

#include "soundif.h"
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#ifdef SUN_AUDIO
#include <sys/audioio.h>
#include <stropts.h>
#include <sys/conf.h>
#else /* SUN_AUDIO */
#include <sys/soundcard.h>
#include <sys/ioctl.h>
#endif /* SUN_AUDIO */

/* ---------------------------------------------------------------------- */

int soundif_fd = -1;

#ifdef SUN_AUDIO
char soundif_name[256] = "/dev/audio";
#else /* SUN_AUDIO */
char soundif_name[256] = "/dev/dsp";
#endif /* SUN_AUDIO */


/* ---------------------------------------------------------------------- */

void soundif_stop(void)
{
	close(soundif_fd);
}

/* ---------------------------------------------------------------------- */

int soundif_start(int sample_rate, int mode)
{
#ifdef SUN_AUDIO
	audio_info_t audioinfo;
	audio_device_t audiodev;
#else /* SUN_AUDIO */
	int sndparam;
#endif /* SUN_AUDIO */

	fprintf(stderr, "soundif: starting \"%s\"\n", soundif_name);
	if ((soundif_fd = open(soundif_name, mode)) < 0) {
		fprintf(stderr, "soundif: Error, cannot open \"%s\"\n",  
			soundif_name);
		return -1;
	}
#ifdef SUN_AUDIO
	if (ioctl(soundif_fd, AUDIO_GETDEV, &audiodev) == -1) {
		fprintf(stderr, "soundif: Error, cannot get audio dev\n");
		return -1;
	}	
	fprintf(stderr, "soundif: Audio device: name %s, ver %s, config %s\n",
		audiodev.name, audiodev.version, audiodev.config);
	AUDIO_INITINFO(&audioinfo);
	audioinfo.record.sample_rate = sample_rate;
	audioinfo.record.channels = 1;
	audioinfo.record.precision = 16;
	audioinfo.record.encoding = AUDIO_ENCODING_LINEAR;
	audioinfo.record.gain = 0x20;
	audioinfo.record.port = AUDIO_LINE_IN;
	audioinfo.monitor_gain = 0;
	if (ioctl(soundif_fd, AUDIO_SETINFO, &audioinfo) == -1) {
		fprintf(stderr, "soundif: Error, cannot set audio params\n");
		return -1;
	}	
	if (ioctl(soundif_fd, I_FLUSH, FLUSHR) == -1) {
		fprintf(stderr, "soundif: Error, cannot flush\n");
		return -1;
	}
#else /* SUN_AUDIO */
	sndparam = AFMT_S16_LE;	/* we want 16 bits/sample signed */
	/* little endian; works only on little endian systems! */
	if (ioctl(soundif_fd, SNDCTL_DSP_SETFMT, &sndparam) == -1) {
		fprintf(stderr, "soundif: Error, cannot set sample size\n");
		return -1;
	}
	if (sndparam != AFMT_S16_LE) {
		fprintf(stderr, "soundif: Error, cannot set sample size to "
			"16 bits\n");
		return -1;
	}
	sndparam = 0;	/* we want only 1 channel */
	if (ioctl(soundif_fd, SNDCTL_DSP_STEREO, &sndparam) == -1) {
		fprintf(stderr, "soundif: Error, cannot set the channel "
			"number\n");
		return -1;
	}
	if (sndparam != 0) {
		fprintf(stderr, "soundif: Error, cannot set the channel "
			"number to 1\n");
		return -1;
	}
	sndparam = sample_rate; 
	if(ioctl(soundif_fd, SNDCTL_DSP_SPEED, &sndparam) == -1) {
		fprintf(stderr, "soundif: Error, cannot set the sample "
			"rate\n");
		return -1;
	}
	if ((10*abs(sndparam-sample_rate)) > sample_rate) {
		fprintf(stderr, "soundif: Error: Sampling rate is %u, "
			"requested %u\n", sndparam, sample_rate);
		return -1;
	}
	if (sndparam != sample_rate) {
		fprintf(stderr, "soundif: Warning: Sampling rate is %u, "
			"requested %u\n", sndparam, sample_rate);
	}
#if 0
	sndparam = 4;
	if (ioctl(soundif_fd, SOUND_PCM_SUBDIVIDE, &sndparam) == -1) {
		fprintf(stderr, "soundif: Warning, subdivide failed, "
			"tx/rx switching will be slow\n");
	}
	if (sndparam != 4) {
		fprintf(stderr, "soundif: Warning, subdivide returned %u\n",
			sndparam);
	}
#endif
#endif /* SUN_AUDIO */
	return 0;
}

/* ---------------------------------------------------------------------- */
