/*
 *      xgraphif.c -- invoke xgraph to display traces
 *
 *	Copyright (C) 1996  
 *          Thomas Sailer (sailer@ife.ee.ethz.ch, hb9jnx@hb9w.che.eu)
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* ---------------------------------------------------------------------- */

#include "xgraphif.h"
#include <sys/types.h>
#include <unistd.h>
#include <signal.h>
#include <stdio.h>
#include <errno.h>

/* ---------------------------------------------------------------------- */

#define XGRAPH_CMDNAME "xgraph"

/* ---------------------------------------------------------------------- */

int xgraph_fd = -1;
static int first_set = 1;

/* ---------------------------------------------------------------------- */

int xgraph_open(const char *argv[])
{
	int pid;
	int fds[2];

	if (xgraph_fd >= 0)
		return -1;
	signal(SIGCHLD, SIG_IGN);
	if (pipe(fds)) {
		perror("xgraph_if");
		return -1;
	}
	if (!(pid = fork())) {
		static const char *dflt_argv[] = { XGRAPH_CMDNAME, NULL };
		/* we're in the child */
		close(0); /* close stdin */
		if (dup2(fds[0],0)) {
			fprintf(stderr, "xgraph_if: dup2 failed\n");
		}
		close(fds[0]);
		close(fds[1]);
		execvp(XGRAPH_CMDNAME, (argv != NULL) ? argv : dflt_argv);
		/* we return only if the exec failed - in this case */
		/* print an error and kill child */
		perror("xgraph");
		exit(-1);
	}
	close(fds[0]);
	if (pid < 0) {
		close(fds[1]);
		xgraph_fd = -1;
		return -1;
	}
	xgraph_fd = fds[1];
	first_set = 1;
	return 0;
}

/* ---------------------------------------------------------------------- */

void xgraph_close(void)
{
	if (write(xgraph_fd, "\n", 1) < 0)
		perror("xgraph_if");
	if (xgraph_fd >= 0) {
		close(xgraph_fd);
	}
	xgraph_fd = -1;
}

/* ---------------------------------------------------------------------- */

void xgraph_newset(const char *setname)
{
	char buf[64];
	int len;

	if (xgraph_fd < 0)
		return;
	if (!first_set)
		if (write(xgraph_fd, "\n", 1) < 0)
			perror("xgraph_if");
	first_set = 0;
	if (setname) {
		len = sprintf(buf, "\"%.62s\"\n", setname);
		if (write(xgraph_fd, buf, len) < 0)
			perror("xgraph_if");
	}
}

/* ---------------------------------------------------------------------- */

void xgraph_newdata(int mode, float x, float y)
{
	static const char *modestr[] = { "", "move ", "draw " };
	char buf[128];
	int len;

	if (xgraph_fd < 0)
		return;
	first_set = 0;
	if (mode < 0 || mode > 2) 
		mode = 0;
	len = sprintf(buf, "%s%f %f\n", modestr[mode], x, y);
	if (write(xgraph_fd, buf, len) < 0)
		perror("xgraph_if");
}

/* ---------------------------------------------------------------------- */

void xgraph_settitle(const char *title)
{
	char buf[128];
	int len;

	if (xgraph_fd < 0)
		return;
	if (title) {
		len = sprintf(buf, "TitleText: %.100s\n", title);
		if (write(xgraph_fd, buf, len) < 0)
			perror("xgraph_if");
	}
}

/* ---------------------------------------------------------------------- */
