;Control U6239 (and similar) synth chip via RS232 interface

;RS232 connected at start up / reset, goes into prog mode:
;Prompt G/D/E/C/W
;   G   -   Go to operational mode with any changes to data entered
;   D   -   Display current EEPROM contents
;   E   -   Enter new frequency data 4 ASCII hex characters
;   C   -   Enter new control register data, 2 ASCII hex characters
;   W   -   Write current settings to EEPROM and go to operational mode
;


    PROCESSOR 16C84
    INCLUDE P16C84.INC
    errorlevel 2    ;Errors only


    #define     LED     PORTA , 1
    #define     SCL     PORTA , 2
    #define     SDA     PORTA , 3

    #define     TXD     PORTB , 7       ;Serial data INPUT FROM PC
    #define     RXD     PORTB , 6       ;   "    "   OUTPUT TO PC
                  
    #define     AckFlag     Flags , 0

    FREQDATA    =   d'10370'    ;1296.25
    CPTestbits  = b'11000110' ; High I, Normal Mode, RDiv = 512, VCap En.


 org 0x2100

    de high(FREQDATA), low(FREQDATA)  ;EE locations 0 / 1 contain frequency info.
    de CPTestbits


    AddrByte    = b'11000010' ;Programmable address 'always valid'  R/W =0

    ;Fout = FREQDATA * 16 * Fref / Rdiv

    __config    0x3ff1  ;CP Off, PUT Dis, WDT Dis, XT

;code starts here

 org     0
    clrw
    movwf   INTCON      ;disable interrupts
    goto    Startup     ;jump to main code
 org 4
ints    retfie          ;interrupt service routine code

Startup bsf     STATUS,RP0      ;ram page 1
    movlw   b'00000000'     ;
    movwf   PORTA       ;
    movlw   b'10000000'     ;B7 Serial data in
    movwf   PORTB
    bcf     STATUS,RP0      ; 0

    bsf     SCL
    bsf     SDA
    bcf     LED

    movlw   0               ;Get saved data in D1/D0/CReg in case
    call    GetEE           ;  ProgMode is called but nothing done
    movwf   D1
    movlw   1
    call    GetEE
    movwf   D0
    movlw   2
    call    GetEE
    movwf   CReg
    bcf     D1, 7           ;Just in case of spurious EE contents
    bsf     CReg, 7


    btfss   TXD         ;If TXD Low at start up, RS232 connected
    goto    ProgMode    ; so go straight into prog mode

MainStart
    bsf     LED         ;Do some LED flashing
    call    LongDelay
    bcf     LED
    call    LongDelay
    bsf     LED
    call    LongDelay
    bcf     LED
    call    LongDelay
    bsf     LED
    call    LongDelay
    bcf     LED
    call    LongDelay
    bsf     LED
    call    LongDelay
    bcf     LED
    call    LongDelay

    movlw   0
    movwf   Preg

    call    SendSynth

    call    LongDelay
    call    LongDelay
    call    LongDelay
    call    LongDelay
    call    LongDelay
    call    LongDelay
    call    LongDelay
    call    LongDelay
    call    LongDelay
    call    LongDelay

TestLock            ;And now just sit monitoring In-Lock flag
    call    ReadSynth
    btfsc   SynthReg , 6    ;In-Lock flag
    bsf     LED
    btfss   SynthReg , 6
    bcf     LED
    call    LongDelay
    goto    TestLock
;.................................
ProgMode
    call    LongDelay
    btfsc   TXD             ;If still low after 100ms delay then OK
    goto    MainStart

    bsf     LED
    movlw   "6"
    call    Send232
    movlw   "2"
    call    Send232
    movlw   "3"
    call    Send232
    movlw   "9"
    call    Send232
    movlw   " "
    call    Send232
    movlw   "C"
    call    Send232
    movlw   "o"
    call    Send232
    movlw   "n"
    call    Send232
    movlw   "t"
    call    Send232
    movlw   "r"
    call    Send232
    movlw   "o"
    call    Send232
    movlw   "l"
    call    Send232
    movlw   0x0D
    call    Send232
    movlw   0x0A
    call    Send232

ProgDoLoop
    call    ClearStack      ;Prevent any spurious commands being generated
    bcf     LED

    movlw   0x0D
    call    Send232
    movlw   0x0A
    call    Send232
    movlw   "G"
    call    Send232
    movlw   "/"
    call    Send232
    movlw   "D"
    call    Send232
    movlw   "/"
    call    Send232
    movlw   "E"
    call    Send232
    movlw   "/"
    call    Send232
    movlw   "C"
    call    Send232
    movlw   "/"
    call    Send232
    movlw   "W"
    call    Send232
    movlw   0x0D
    call    Send232
    movlw   0x0A
    call    Send232

    call    WaitData        ;Char pushed onto stack in Waitdata
    bsf     LED             ;Flash the LED whenever a command is entered
                           
    movlw   "G"             ;G command starts going with new freq
    subwf   Stack1, W
    btfsc   STATUS, Z
    goto    MainStart       ;D1/0/Creg  may or may not have been updated

    movlw   "E"             
    subwf   Stack1, W       
    btfsc   STATUS, Z       
    goto    ECommand

    movlw   "W"             ;Writes new data to EEPROM and starts synth
    subwf   Stack1, W
    btfsc   STATUS, Z
    goto    WriteData 

    movlw   "D"             ;Display current EEprom data
    subwf   Stack1, W
    btfsc   STATUS, Z
    goto    ShowData

    movlw   "C"             ;Enter latest C data
    subwf   Stack1, W
    btfsc   STATUS, Z
    goto    CCommand

    movlw   "~"             ;Any other illegal character
    call    Send232
    movlw   0x0D
    call    Send232
    movlw   0x0A
    call    Send232

    goto    ProgDoLoop      ;Any other character
;.....................
WriteData                   ; 'W' Write data in D1/D0 to EEPROM
    movf    D1 , W
    movwf   EEDATA
    movlw   0
    movwf   EEADR
    call    StoreEE

    movf    D0 , W
    movwf   EEDATA
    movlw   1
    movwf   EEADR
    call    StoreEE

    movf    CReg , W
    movwf   EEDATA
    movlw   2
    movwf   EEADR
    call    StoreEE

    movlw   "E"
    call    Send232
    movlw   "E"
    call    Send232
    movlw   " "
    call    Send232
    movlw   "W"
    call    Send232
    movlw   "r"
    call    Send232
    movlw   "i"
    call    Send232
    movlw   "t"
    call    Send232
    movlw   "t"
    call    Send232
    movlw   "e"
    call    Send232
    movlw   "n"
    call    Send232
    movlw   0x0D
    call    Send232
    movlw   0x0A
    call    Send232

    goto    ProgDoLoop
;....................
ECommand                    ;Extract 4 ascii characters from stack and
    movlw   "D"             ;  update D1/0, but don't write to EE
    call    Send232
    movlw   "a"
    call    Send232
    movlw   "t"
    call    Send232
    movlw   "a"
    call    Send232
    movlw   "?"
    call    Send232
    movlw   0x0D
    call    Send232
    movlw   0x0A
    call    Send232

ELoop
    call    WaitData        ;Currently Stack1 contains "E"
    movf    Stack1 , W
    call    Send232         ;Echo it back

    movlw   "E"             ;Collect characters until this is shifted  
    subwf   Stack5, W       ;  and eventually appears in Stack5 
    btfsc   STATUS, Z       
    goto    EDone
    goto    ELoop
;.......
EDone
    movf    Stack4 , W      ;  convert to binary in D1/0, then send back
    call    DecodeHex       ;MS Nibble
    movwf   D1              ;Stack 6/5/4/3/2/1
    swapf   D1              ;        E x x x x

    movf    Stack3 , W
    call    DecodeHex
    addwf   D1
    bcf     D1, 7           ;Mask off any MSB entered in error

    movf    Stack2 , W
    call    DecodeHex
    movwf   D0
    swapf   D0

    movf    Stack1 , W      ;LSNibble
    call    DecodeHex
    addwf   D0

    movlw   0x0D
    call    Send232
    movlw   0x0A
    call    Send232

    movlw   "E"
    call    Send232
    swapf   D1 , W          ;Return data on serial link
    andlw   0x0F
    call    SendHex
    movf    D1 , W
    andlw   0x0F
    call    SendHex
    swapf   D0 , W         
    andlw   0x0F
    call    SendHex
    movf    D0 , W
    andlw   0x0F
    call    SendHex
    movlw   0x0D
    call    Send232
    movlw   0x0A
    call    Send232         ;D1/0 updated, but not written to EE
  
    goto    ProgDoLoop
;............................
CCommand                    ;Extract 4 ascii characters from stack and
    movlw   "D"             ;  update D1/0, but don't write to EE
    call    Send232
    movlw   "a"
    call    Send232
    movlw   "t"
    call    Send232
    movlw   "a"
    call    Send232
    movlw   "?"
    call    Send232
    movlw   0x0D
    call    Send232
    movlw   0x0A
    call    Send232

CLoop
    call    WaitData        ;Currently Stack1 contains "C"
    movf    Stack1 , W
    call    Send232         ;Echo it back

    movlw   "C"             ;Collect characters until this is shifted  
    subwf   Stack3, W       ;  and eventually appears in Stack5 
    btfsc   STATUS, Z       
    goto    CDone           ;This will continue looping until any Exxxx
    goto    CLoop           ;  appears in the input data
;.......
CDone
    movf    Stack2 , W      ;  convert to binary in D1/0, then send back
    call    DecodeHex       ;MS Nibble
    movwf   CReg            ;Stack 6/5/4/3/2/1
    swapf   CReg            ;            E x x

    movf    Stack1 , W
    call    DecodeHex
    addwf   CReg
    bsf     CReg , 7        ;MSB must always be 1

    movlw   0x0D
    call    Send232
    movlw   0x0A
    call    Send232
    movlw   "C"
    call    Send232
    movlw   "-"
    call    Send232
    swapf   CReg , W        ;Return data on serial link
    andlw   0x0F
    call    SendHex
    movf    CReg , W
    andlw   0x0F
    call    SendHex
    movlw   0x0D
    call    Send232
    movlw   0x0A
    call    Send232         ;CReg updated, but not written to EE
  
    goto    ProgDoLoop
;............................
ShowData                    ;Send Contents of EE to RS232

    movlw   "D"
    call    Send232
    movlw   "-"
    call    Send232

    movlw   0
    call    GetEE
    movwf   RxChar          ;Use as temporary storage as Temp is used below
    swapf   RxChar, W
    andlw   0x0F
    call    SendHex
    movf    RxChar, W
    andlw   0x0F
    call    SendHex

    movlw   1
    call    GetEE
    movwf   RxChar          ;Use as temporary storage as Temp is used below
    swapf   RxChar, W
    andlw   0x0F
    call    SendHex
    movf    RxChar, W
    andlw   0x0F
    call    SendHex

    movlw   0x0D
    call    Send232
    movlw   0x0A
    call    Send232

    movlw   "C"
    call    Send232
    movlw   "-"
    call    Send232

    movlw   2
    call    GetEE
    movwf   RxChar          ;Use as temporary storage as Temp is used below
    swapf   RxChar, W
    andlw   0x0F
    call    SendHex
    movf    RxChar, W
    andlw   0x0F
    call    SendHex

    movlw   0x0D
    call    Send232
    movlw   0x0A
    call    Send232
  
    goto    ProgDoLoop
;================================================
WaitData              ;Sit in here waiting for RS232 start bit
    btfss   TXD             ;look for start transition, low/high = 1/0
    goto    WaitData        ;
    bcf     LED             ;Blink LED for each character entered
                            
    movlw   d'80'      
    movwf   DelCount        ;
    nop
LoopStartBit                ;this loop 5.N - 1 long = 
    nop
    nop
    decfsz  DelCount
    goto    LoopStartBit

    btfss   TXD             ;make sure it's still high, now centre of bit
    goto    WaitData        ;
    movlw   8               ;
    movwf   Counter
    clrf    RxChar          ; to here from transition = half bit
ByteLoop
    call    BitDelay
    nop
    bcf     STATUS , C     
    btfss   TXD         
    bsf     STATUS , C     
    rrf     RxChar        
    decfsz  Counter
    goto    ByteLoop        ;this loop BitDelay + 8
    movf    RxChar , w      ;Returns with stop bit+  slack period

    movlw   "a"             ;Convert to Upper case
    subwf   RxChar, W       ;W = Char - "a"
    btfss   STATUS, C       ;if-ve (C=0) char < "a" so ignore
    goto    UpCaseDone

    movf    RxChar, W
    sublw   "z"             ;W = "z" - Char
    btfss   STATUS, C       ;if -ve (C=0)  char > "z" so ignore
    goto    UpCaseDone

    bcf     RxChar, 5       ;Reset bit 5 to force upper case
UpCaseDone
    movf    RxChar, W
    call    PushStack       ;RxChar >> Stack1, previous 5 chars move down one
    bsf     LED
    return                  ;received byte in RxChar and W
;-------------------------
BitDelay                ;For RS232 1200 Baud link
    movlw   d'164'          ; delay = 5.N + 5  = 825us
    movwf   DelCount
HalfBaudDelLoop
    nop
    nop
    decfsz  DelCount
    goto    HalfBaudDelLoop
    return
;---------------------------
SendHex                     ;Send a binary nibble as ASCII hex data
    addlw   0x30
    movwf   Temp
    sublw   0x39        ;W = "9" - W, if -ve, A-F, C = 0 so add 7
    movf    Temp, W     ;Recover data,  doesn't affect Carry
    btfss   STATUS , C
    addlw   7

Send232             ;Enter with data in W, send it to PC
    movwf   Temp
    bsf     RXD         ;start bit
    call    BitDelay    ;
    movlw   8
    movwf   Counter
    nop
    nop
SendDatLoop
    rrf     Temp        ;bit into C
    btfsc   STATUS , C
    bcf     RXD
    btfss   STATUS , C
    bsf     RXD         ;could have 2us jitter depending on 1/0
    call    BitDelay
    decfsz  Counter
    goto    SendDatLoop     ;loop 8 + BitDelay long
    bcf     RXD         ;stop bit
    call    BitDelay    
    call    BitDelay    ;Two stop bits
    return
;-------------------------
UpCase
    return

;------------------------------------------------
ReadSynth
    call    I2CStart
    movlw   AddrByte
    iorlw   1           ;Set Read mode
    call    I2CSend     ;Send Address
    call    I2CReceive      ;Read data, stored in SynthReg
    call    I2CStop
    return
;------------------------------------------------
SendSynth       
    call    I2CStart
    movlw   AddrByte    ;Bit 0 already set to 0 for write
    call    I2CSend
    movf    D1, W       ;MS Frequency info
    call    I2CSend
    movf    D0, W       ;LS Frequency Info
    call    I2CSend
    movf    CReg, W
    call    I2CSend
    movf    Preg, W
    call    I2CSend
    call    I2CStop
    return
;------------------------------------------------
I2CStart        ;High to low of SDA while SCL high
    bsf     SCL         ;Just in case, includes a latent Stop if wrong
    bsf     SDA         ;Will be high already if set as input,
                ;  ensures no change on setting as output
    bsf     STATUS, RP0     ;
    bcf     SDA         ;Set SDA line as output
    bcf     STATUS, RP0
    bcf     SDA         ;This is the start condition
    nop
    bcf     SCL         ;This falling edge initiates first data bit
    return          ;Exit with SDA Low set as output, SCL Low
;--------------------
I2CSend         ;Enter with byte in W
    movwf   Temp    ;Uses Registers - Temp, Counter, Flags -  AckFlag
    movlw   8           
    movwf   Counter
    bcf     SCL         ;SCL low just in case

    bsf     STATUS, RP0
    bcf     SDA         ;Set SDA line as output as we don't
    bcf     STATUS, RP0     ;  know its history
I2CTxLoop
    rlf     Temp
    btfsc   STATUS, C
    bsf     SDA
    btfss   STATUS, C
    bcf     SDA         ;Send data MSB first
    nop
    bsf     SCL         ;Clock it in
    nop
    bcf     SCL
    decfsz  Counter
    goto    I2CTxLoop       
    bsf     SDA         ;Exit loop with SCL low, SDA high
    
    bsf     STATUS, RP0
    bsf     SDA         ;Set SDA line as input
    bcf     STATUS, RP0

    bsf     SCL         ;Clock in the Ack data
    bcf     AckFlag
    btfsc   SDA
    bsf     AckFlag     ;Read ACK from slave device into AckFlag
    bcf     SCL         ;Exit with SCL Low and SDA set as INPUT
    return          ;  ready for subsequent read operation
;--------------------
I2CReceive          ;Enter with SDA set as input, SCL should be low, but...
    bcf     SCL         ;Ack data in AckFlag
    movlw   8           
    movwf   Counter
    clrf    SynthReg
I2CRxLoop
    bsf     SCL         ;Data should be stable after clock high edge
    
    bcf     STATUS, C
    btfsc   SDA
    bsf     STATUS, C
    rlf     SynthReg    ;Rotate 8 data bits into SynthReg
    bcf     SCL         ;Initiates next data bit

    decfsz  Counter
    goto    I2CRxLoop       ;Exit loop with SCL low, SDA input / unknown

    bcf     SDA         ;While set as input, does not affect the signal
    btfsc   AckFlag
    bsf     SDA
    bsf     STATUS, RP0
    bcf     SDA         ;Set SDA line as output
    bcf     STATUS, RP0     ;This will output the Ack data

    bsf     SCL         ;Send Ack to slave device
    nop
    bcf     SCL

    bsf     STATUS, RP0
    bsf     SDA         ;Set SDA line as input again
    bcf     STATUS, RP0

    return          ;Exit with SCL Low, SDA Input, data in SynthReg
;--------------------
I2CStop         ;Low to high of SDA with SCL high 
    bcf     SCL         ;Should already be low, but...
    
    bcf     SDA         
    bsf     STATUS, RP0
    bcf     SDA         ;Set SDA line as output again
    bcf     STATUS, RP0

    bsf     SCL
    nop
    bsf     SDA         ;This is the Stop condition

    return          ;Exit with both lines high, SDA as output
;--------------------
LongDelay           ;Approx 100ms 
    movlw   d'78'
    movwf   DelCount2
LongDelLoop
    call    Delay
    decfsz  DelCount2
    goto    LongDelLoop
    return
;--------------------
Delay       ;delay of 256 clocks, Approx 1.285ms at Fc = 1MHz
    clrf   DelCount
DelLoop
    nop
    nop
    decfsz  DelCount
    goto    DelLoop
    return
;--------------------
GetEE
    movwf   EEADR
    bsf     STATUS,RP0      ;ram page 1
    bsf     EECON1 , RD
    bcf     STATUS , RP0    ;ram page 0
    movf    EEDATA , W
    return
;-------------------------
StoreEE         ;writes the data byte in Data reg to EEprom EEADR
    bsf     STATUS , RP0
    bsf     EECON1 , WREN       ;Enable EEprom writing
    movlw   0x55
    movwf   EECON2
    movlw   0xAA
    movwf   EECON2
    bsf     EECON1 , WR
EEwaitW
    btfss   EECON1 , EEIF
    goto    EEwaitW
    bcf     EECON1 , EEIF       ;has to be cleared manually
    bcf     EECON1 , WR
    bcf     STATUS , RP0
    
    return
;-------------------------
DecodeHex           ;Single ASCII character to binary 
    movwf   Temp            ;Uses Temp
    movlw   0x30
    subwf   Temp            ;subtract $30 and leave in Temp
    btfss   STATUS , C      ;if < $30, result -ve so C=0, error, set=0
    clrf    Temp

    movf    Temp , W        
    sublw   d'9'            ;W = 9 - W
    btfsc   STATUS , C      ;if -ve, C=0, char > 9, so subtract 7
    goto    ConvDone
    movlw   d'7'
    subwf   Temp            ;
ConvDone
    movf    Temp , W        ;
    andlw   0x0F            ;mask out any rubbish left in
    return              
;-------------------------
ClearStack
    clrf    Stack1
    clrf    Stack2
    clrf    Stack3
    clrf    Stack4
    clrf    Stack5
    clrf    Stack6
    return    
;-------------------------

PushStack
    movf    Stack5 , W
    movwf   Stack6              ;Stack6 contains oldest data
    movf    Stack4 , W
    movwf   Stack5
    movf    Stack3 , W
    movwf   Stack4
    movf    Stack2 , W
    movwf   Stack3
    movf    Stack1 , W
    movwf   Stack2
    movf    RxChar , W
    movwf   Stack1
    return
;-------------------------
    cblock 0x10
        D0
        D1
        SynthReg
        Preg
        CReg
        Counter
        RxChar
        DelCount    
        DelCount1       
        DelCount2      
        Temp       
        BitCount
        Flags
        Stack1
        Stack2
        Stack3
        Stack4
        Stack5
        Stack6
    endc

    end
