/*
  dsniff.c

  simple libnids-based sniffer, because DrHoney wanted one.

  decodes FTP, Telnet, HTTP, POP, IMAP, SNMP, Rlogin, NFS, X11 auth info.
  this is for demonstration purposes and educational use only.
  
  Copyright (c) 1999 Dug Song <dugsong@monkey.org>
  All rights reserved, all wrongs reversed.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
  3. The name of author may not be used to endorse or promote products
     derived from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
  ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  $Id: dsniff.c,v 1.21 2000/01/21 08:21:17 dugsong Exp $
*/

#include "config.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/udp.h>
#include <arpa/inet.h>
#include <arpa/nameser.h>
#include <arpa/telnet.h>
#include <rpc/rpc.h>
#ifdef HAVE_RPC_RPCENT_H
#include <rpc/rpcent.h>
#endif
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#ifdef HAVE_DB_185_H
#include <db_185.h>
#elif HAVE_DB_H
#include <db.h>
#endif
#include <md5.h>
#include <nids.h>

#include "version.h"

#define MAX_LINES	6
#define MIN_SNAPLEN	512
#define PMAPSIZE	64
#define XIDMAPSIZE	64
#define PROG_PMAP	100000
#define PROG_MOUNT	100005

#define FRAGLEN(x)	(x & 0x7fffffff)
#define LASTFRAG(x)	(x & (1 << 31))

struct pmap_entry {
  u_long dst;
  u_short dport;
  int prog;
};

struct xid_map_entry {
  u_long xid;
  u_long src;
  u_long dst;
  int prog;
  char *data;
};

struct rpc_record {
  u_long rm;
  u_char *data;
  int len;
};

struct rpc_call {
  int xid;
  int prog;
  int proc;
  u_char *cred;
  u_char *verf;
  u_char *args;
};

struct rpc_reply { /* accepted, successful only */
  int xid;
  u_char *verf;
  u_char *results;
};

struct dsniff_rec {
  u_long src;
  u_long dst;
  u_int svc;
  time_t time;
  char *data;
  int len;
};

extern int base64_decode(char *in, char *out, int len);
extern u_char *libnet_host_lookup(u_long in, u_short use_name);

/* Options. */
int	Opt_dns = 1;
int	Opt_read = 0;
int	Opt_write = 0;

/* Globals. */
DB     *db;
char	Buf[BUFSIZ];
int	Lines = MAX_LINES;
int	Snaplen = MIN_SNAPLEN;
int	pmap_next = 0;
int	pmap_hint = 0;
int	xid_map_next = 0;
int	xid_map_hint = 0;
struct pmap_entry pmap_map[PMAPSIZE];
struct xid_map_entry xid_map[XIDMAPSIZE];

void
usage(void)
{
  fprintf(stderr, "Usage: dsniff [-n] [-i interface] [-r|-w file]\n");
  exit(1);
}

#ifndef OpenBSD
size_t
strlcpy(char *dst, const char *src, size_t siz)
{
  char *d = dst;
  const char *s = src;
  size_t n = siz;
  
  if (n != 0 && --n != 0) {
    do {
      if ((*d++ = *s++) == 0) break;
    } while (--n != 0);
  }
  if (n == 0) {
    if (siz != 0) *d = '\0';
    while (*s++) ;
  }
  return (s - src - 1);
}

size_t
strlcat(char *dst, const char *src, size_t siz)
{
  char *d = dst;
  const char *s = src;
  size_t dlen, n = siz;

  while (*d != '\0' && n-- != 0) d++;
  dlen = d - dst;
  n = siz - dlen;
  
  if (n == 0) return (dlen + strlen(s));
  for ( ; *s != '\0'; s++) {
    if (n != 1) {
      *d++ = *s;
      n--;
    }
  }
  *d = '\0';
  
  return (dlen + (s - src));
}
#endif /* !OpenBSD */

#ifdef DEBUG
/* Print a buffer in hex. */
void
hex_print(u_char *buf, int len)
{
  int i;
  
  for (i = 0; i < len; i++) {
    if (!(i & 0xF)) printf("\n");
    else if (!(i % 2)) printf(" ");
    printf("%.2x", buf[i]);
  }
  printf("\n");
}
#endif /* DEBUG */

/* Locate substring in a binary string. */
u_char *
bufbuf(u_char *big, int blen, u_char *little, int llen)
{
  u_char *p;

  for (p = big; p <= big + blen - llen; p++) {
    if (memcmp(p, little, llen) == 0)
      return (p);
  }
  return (NULL);
}
  
/* XXX - seriously lame portmap cache. ugh. */
void
pmap_enter(u_long dst, u_short dport)
{
  struct pmap_entry *pp;

  pp = &pmap_map[pmap_next];
  pp->dst = dst;
  pp->dport = dport;

  if (++pmap_next >= PMAPSIZE)
    pmap_next = 0;
}

int
pmap_find(u_long dst, u_short dport)
{
  struct pmap_entry *pp;
  int i;
  
  if (dport == 111)
    return (1);
  
  i = pmap_hint;
  do {
    pp = &pmap_map[i];
    if (pp->dst == dst && pp->dport == dport) {
      /* match */
      pmap_hint = i;
      return (1);
    }
    if (++i >= PMAPSIZE)
      i = 0;
  } while (i != pmap_hint);
  
  return (0);
}

/* xid_map borrowed from tcpdump's print-nfs.c */
void
xid_map_enter(u_long src, u_long dst, int xid, int prog, u_char *data)
{
  struct xid_map_entry *mp;
  
  mp = &xid_map[xid_map_next];
  
  if (++xid_map_next >= XIDMAPSIZE)
    xid_map_next = 0;
  
  mp->xid = xid;
  mp->src = src;
  mp->dst = dst;
  mp->prog = prog;
  mp->data = data;
}

struct xid_map_entry *
xid_map_find(u_long src, u_long dst, int xid)
{
  struct xid_map_entry *mp;
  int i;

  /* Start searching from where we last left off. */
  i = xid_map_hint;
  do {
    mp = &xid_map[i];
    if (mp->xid == xid && mp->src == src && mp->dst == dst) {
      /* match */
      xid_map_hint = i;
      return (mp);
    }
    if (++i >= XIDMAPSIZE)
      i = 0;
  } while (i != xid_map_hint);
  
  return (NULL);
}

DBT *
db_md5_key(struct dsniff_rec *d)
{
  static DBT key;
  static u_char hash[16];
  MD5_CTX ctx;

  MD5Init(&ctx);
  MD5Update(&ctx, (u_char *)&d->src, sizeof(d->src));
  MD5Update(&ctx, (u_char *)&d->dst, sizeof(d->dst));
  MD5Update(&ctx, (u_char *)&d->svc, sizeof(d->svc));
  MD5Update(&ctx, d->data, d->len);
  MD5Final(hash, &ctx);

  key.data = hash;
  key.size = sizeof(hash);

  return (&key);
}

char *
svc_name(u_int num)
{
  static char noname[16];
  struct servent *sp;
  struct rpcent *rp;
  char *name = "unknown";
  
  if (num < 1024) {
    if ((sp = getservbyport(htons(num), "tcp")) != NULL ||
	(sp = getservbyport(htons(num), "udp")) != NULL)
      name = sp->s_name;
    else {
      snprintf(noname, sizeof(noname), "port %d", num);
      name = noname;
    }
  }
  else if (num >= 100000) {
    if ((rp = getrpcbynumber(num)) != NULL)
      name = rp->r_name;
    else {
      snprintf(noname, sizeof(noname), "prog %d", num);
      name = noname;
    }
  }
  else if (num >= 6000 && num <= 6010) {
    name = "x11";
  }
  return (name);
}

/* Strip telnet options, as well as suboption data. */
int
strip_telopts(u_char *buf, int len)
{
  int i, j, subopt = 0;
  char *p, *q;
  
  for (i = j = 0; i < len; i++) {
    if (buf[i] == IAC) {
      if (++i >= len) break;
      else if (buf[i] > SB)
	i++;
      else if (buf[i] == SB) {
	/* XXX - check for autologin username. */
	p = buf + i + 1;
	if ((q = bufbuf(p, len - i, "\xff", 1)) != NULL) {
	  if ((p = bufbuf(p, q - p, "USER\x01", 5)) != NULL) {
	    p += 5;
	    buf[j++] = '[';
	    memcpy(buf + j, p, q - p); j += q - p;
	    buf[j++] = ']'; buf[j++] = '\n';
	  }
	}
	subopt = 1;
      }
      else if (buf[i] == SE) {
	if (!subopt) j = 0;
	subopt = 0;
      }
    }
    else if (!subopt) {
      /* XXX - convert isolated carriage returns to newlines. */
      if (buf[i] == '\r' && i + 1 < len && buf[i + 1] != '\n')
	buf[j++] = '\n';
      else
	buf[j++] = buf[i];
    }
  }
  buf[j] = '\0';
  
  return (j);
}

/* Strip a string buffer down to a maximum number of lines. */
int
strip_lines(char *buf, int max_lines)
{
  char *p;
  int lines, nonascii;

  if (!buf) return (0);

  lines = nonascii = 0;
  
  for (p = buf; *p && lines < max_lines; p++) {
    if (*p == '\n') lines++;
    if (!isascii(*p)) nonascii++;
  }
  if (*p) *p = '\0';

  /* XXX - lame ciphertext heuristic */
  if (nonascii * 3 > p - buf)
    return (0);

  return (lines);
}

int
get_asn1_len(u_char **buf)
{
  u_char *p;
  int len, num = 0;
  
  p = *buf;
  
  if (*p >= 128) {	/* Long form */
    len = *p++ & ~128;
    
    if (len == 1) {
      num = *p++;
    }
    else if (len == 2) {
      GETSHORT(num, p);
    }
    else if (len == 3) {
      p--; GETLONG(num, p);
      num &= 0xFFF;
    }
    else if (len== 4)
      GETLONG(num, p);
  }
  else num = *p++;	/* Short form */

  *buf = p;
  
  return (num);
}

int
decode_ftp_auth(char *buf, int len)
{
  char *p;
  
  Buf[0] = '\0';

  for (p = strtok(buf, "\r\n"); p != NULL; p = strtok(NULL, "\r\n")) {
    if (strncasecmp(p, "USER ", 5) == 0 ||
	strncasecmp(p, "PASS ", 5) == 0 ||
	strncasecmp(p, "ACCT ", 5) == 0) {
      strlcat(Buf, p, sizeof(Buf));
      strlcat(Buf, "\n", sizeof(Buf));
    }
  }
  return (strlen(Buf));
}

int
decode_http_auth(char *buf, int len)
{
  char *p, *s, *e;
  int i;

  Buf[0] = '\0';
  
  /* Process requests. */
  for (s = buf; (e = bufbuf(s, len, "\r\n\r\n", 4)) != NULL; s = e + 4) {
    len -= (e + 4) - s;
    *e = '\0';
    
    /* Check for auth info. */
    if ((p = strstr(s, "uthorization: ")) == NULL)
      continue;

    /* Process header. */
    for (p = strtok(s, "\r\n"); p != NULL; p = strtok(NULL, "\r\n")) {
      if (!strlen(p)) {
	continue;
      }
      else if (strncasecmp(p, "GET ", 4) == 0 ||
	       strncasecmp(p, "POST ", 5) == 0) {
	strlcat(Buf, p, sizeof(Buf));
	strlcat(Buf, "\n", sizeof(Buf));
      }
      else if (strncasecmp(p, "Host: ", 6) == 0) {
	strlcat(Buf, p, sizeof(Buf));
	strlcat(Buf, "\n", sizeof(Buf));
      }
      else if (strncasecmp(p, "Authorization: Basic ", 21) == 0) {
	strlcat(Buf, p, sizeof(Buf));
	p += 21;
	i = base64_decode(p, p, strlen(p));
	p[i] = '\0';
	strlcat(Buf, " [", sizeof(Buf));
	strlcat(Buf, p, sizeof(Buf));
	strlcat(Buf, "]\n", sizeof(Buf));
      }
    }
  }
  return (strlen(Buf));
}

int
decode_pop_auth(char *buf, int len)
{
  char *p;
  int i;
  
  Buf[0] = '\0';

  for (p = strtok(buf, "\r\n"); p != NULL; p = strtok(NULL, "\r\n")) {
    if (strncasecmp(p, "AUTH PLAIN", 10) == 0 ||
	strncasecmp(p, "AUTH LOGIN", 10) == 0) {
      char *user, *pass;
      
      /* Decode SASL auth. */
      strlcat(Buf, p, sizeof(Buf));
      strlcat(Buf, "\n", sizeof(Buf));

      if ((user = strtok(NULL, "\r\n")) != NULL &&
	  (pass = strtok(NULL, "\r\n")) != NULL) {
	strlcat(Buf, user, sizeof(Buf));
	i = base64_decode(user, user, strlen(user));
	user[i] = '\0';
	strlcat(Buf, " [", sizeof(Buf));
	strlcat(Buf, user, sizeof(Buf));
	strlcat(Buf, "]\n", sizeof(Buf));
	
	strlcat(Buf, pass, sizeof(Buf));
	i = base64_decode(pass, pass, strlen(pass));
	pass[i] = '\0';
	strlcat(Buf, " [", sizeof(Buf));
	strlcat(Buf, pass, sizeof(Buf));
	strlcat(Buf, "]\n", sizeof(Buf));
      }
    }
    /* Save regular POP2, POP3 auth info. */
    else if (strncasecmp(p, "USER ", 5) == 0 ||
	     strncasecmp(p, "PASS ", 5) == 0 ||
	     strncasecmp(p, "HELO ", 5) == 0) {
      strlcat(Buf, p, sizeof(Buf));
      strlcat(Buf, "\n", sizeof(Buf));
    }
  }
  return (strlen(Buf));
}

int
decode_imap_auth(char *buf, int len)
{
  char *p, *q;
  
  Buf[0] = '\0';

  for (p = strtok(buf, "\r\n"); p != NULL; p = strtok(NULL, "\r\n")) {
    if ((q = strchr(p, ' ')) != NULL) {
      if (strncasecmp(q + 1, "LOGIN ", 6) == 0) {
	strlcat(Buf, p, sizeof(Buf));
	strlcat(Buf, "\n", sizeof(Buf));
      }
    }
  }
  return (strlen(Buf));
}

int
decode_rlogin_auth(char *buf, int len)
{
  char *p, *q;

  p = buf + 1;				/* Skip first NULL */

  strlcpy(Buf, "[", sizeof(Buf));
  strlcat(Buf, p, sizeof(Buf));		/* Local username */
  strlcat(Buf, ":", sizeof(Buf));
  p += strlen(p) + 1;

  strlcat(Buf, p, sizeof(Buf));		/* Remote username */
  strlcat(Buf, "]\n", sizeof(Buf));
  p += strlen(p) + 1;

  p += strlen(p) + 1;			/* Skip term info */

  if ((q = strstr(p, "\xff\xffss")) != NULL)	/* Skip window size */
    p += 12;
  
  for (p = strtok(p, "\r\n"); p != NULL; p = strtok(NULL, "\r\n")) {
    strlcat(Buf, p, sizeof(Buf));
    strlcat(Buf, "\n", sizeof(Buf));
  }
  return (strlen(Buf));
}

/* XXX - lame. */
int
decode_x11_auth(char *buf, int len)
{
  char *p, *q;
  int i;

  p = buf + 12;

  if (strncmp(p, "MIT-MAGIC-COOKIE-1", 18) != 0 || len < 36)
    return (0);
  
  strlcpy(Buf, "MIT-MAGIC-COOKIE-1 ", sizeof(Buf));
  
  p += 20;
  len -= 20;
  q = Buf + 19;
  
  for (i = 0; i < 16 && i < len; i++)
    sprintf(q + (i * 2), "%.2x", (u_char)p[i]);
  strlcat(Buf, "\n", sizeof(Buf));
  
  return (strlen(Buf));
}

int
decode_snmp_auth(u_char *buf, int len)
{
  u_char *p = buf;
  int i, vers;

  if (len < 20) return (0);
  
  if ((*p++ & 0x1F) != 16)	/* XXX - check for sequence */
    return (0);
  i = get_asn1_len(&p);	/* XXX - skip sequence length */
  
  if ((*p++ & 0x1F) != 2)	/* XXX - check for int */
    return (0);
  if (get_asn1_len(&p) != 1)	/* XXX - check version length (always 1?) */
    return (0);
  vers = *p++;

  p++;				/* XXX - skip class, tag. */
  i = get_asn1_len(&p);		/* community string length */
  if ((p - buf) + i > len)
    return (0);
  p[i] = '\0';

  snprintf(Buf, sizeof(Buf), "[version %d]\n%s\n", vers + 1, p);
  
  return (strlen(Buf));
}

/* Decode mount filehandle into nfsshell format. :-) */
int
decode_rpc_mount(char *path, u_char *buf, int len)
{
  int i;
  char fh[128];

  if (len < 32) return (0);
  
  for (i = 0; i < 32; i++) {
    sprintf(fh + (i * 3), "%.2x ", buf[i]);
  }
  fh[95] = '\0';

  return (snprintf(Buf, sizeof(Buf), "%s [%s]\n", path, fh));
}

void
print_rec(struct dsniff_rec *dr)
{
  char tstring[24];
  struct tm *tm;
  char *src, *dst, *svc;

  tm = localtime(&dr->time);
  strftime(tstring, sizeof(tstring), "%x %X", tm);

  src = libnet_host_lookup(dr->src, Opt_dns);
  dst = libnet_host_lookup(dr->dst, Opt_dns);
  svc = svc_name(dr->svc);
  
  printf("-----------------\n");
  printf("%s %s -> %s (%s)\n", tstring, src, dst, svc);
  fwrite(dr->data, 1, dr->len, stdout);
  printf("\n");
  fflush(stdout);
}

void
print_rec_uniq(struct dsniff_rec *dr)
{
  DBT data;

  dr->time = time(NULL);

  data.size = sizeof(*dr) + dr->len;
  
  if ((data.data = malloc(data.size)) == NULL)
    nids_params.no_mem("print_rec");

  /* XXX - who needs portable sniffer logs anyhow? */
  *((struct dsniff_rec *)data.data) = *dr;
  memcpy(data.data + sizeof(*dr), dr->data, dr->len);

  if (db->put(db, db_md5_key(dr), &data, R_NOOVERWRITE) == 0) {
    if (!Opt_write)
      print_rec(dr);
    db->sync(db, 0);
  }
}
  
void
print_db(DB *db)
{
  DBT key, data;
  struct dsniff_rec *dr;

  while (db->seq(db, &key, &data, R_NEXT) == 0) {
    dr = (struct dsniff_rec *)data.data;
    dr->data = data.data + sizeof(*dr);
    print_rec(dr);
  }
}

void
process_udp_client(struct ip *ip, struct udphdr *udp)
{
  struct dsniff_rec dr;
  u_char *udp_b;
  int udp_bl;

  udp_b = (u_char *)(udp + 1);
  udp_bl = ntohs(udp->uh_ulen) - sizeof(*udp);
  
  switch (ntohs(udp->uh_dport)) {
  case 161:
    if ((dr.len = decode_snmp_auth(udp_b, udp_bl)) <= 0)
      return;
    break;
  default:
    return;
    break;
  }
  dr.src = ip->ip_src.s_addr;
  dr.dst = ip->ip_dst.s_addr;
  dr.svc = ntohs(udp->uh_dport);
  dr.data = Buf;

  print_rec_uniq(&dr);
}
  
void
process_tcp_client(struct tcp_stream *ts, int len)
{
  struct dsniff_rec dr;
  
  if (len <= 0) return;

  /* Make subsequent string operations safe. */
  if (ts->server.bufsize > len)
    ts->server.data[len] = '\0';
  
  switch (ts->addr.dest) {
  case 21:
    if (!strip_telopts(ts->server.data, len))
      return;
    if (!(dr.len = decode_ftp_auth(ts->server.data, len)))
      return;
    break;
  case 23:
    if (!strip_telopts(ts->server.data, len))
      return;
    if (!strip_lines(ts->server.data, Lines))
      return;
    if ((dr.len = strlcpy(Buf, ts->server.data, sizeof(Buf))) > sizeof(Buf))
      return;
    break;
  case 80:
    if (!(dr.len = decode_http_auth(ts->server.data, len)))
      return;
    break;
  case 109: case 110:
    if (!(dr.len = decode_pop_auth(ts->server.data, len)))
      return;
    break;
  case 143: case 220:
    if (!(dr.len = decode_imap_auth(ts->server.data, len)))
      return;
    break;
  case 513: case 514:
    if (!decode_rlogin_auth(ts->server.data, len))
      return;
    if (!strip_lines(Buf, Lines))
      return;
    dr.len = strlen(Buf);
    break;
  case 6000: case 6001: case 6002: case 6003: case 6004: case 6005:
    if (!(dr.len = decode_x11_auth(ts->server.data, len)))
      return;
    break;
  default:
    return;
    break;
  }
  dr.src = ts->addr.saddr;
  dr.dst = ts->addr.daddr;
  dr.svc = ts->addr.dest;
  dr.data = Buf;

  print_rec_uniq(&dr);
}

void
process_rpc_call(struct tuple4 *addr, u_char *buf, int len)
{
  struct rpc_msg *rm;
  struct rpc_call rc;
  struct dsniff_rec dr;
  u_char *p;
  int i;

  if (len < 32) return;
  
  rm = (struct rpc_msg *)buf;
  rc.xid = ntohl(rm->rm_xid);
  rc.prog = ntohl(rm->rm_call.cb_prog);
  rc.proc = ntohl(rm->rm_call.cb_proc);
  rc.cred = (u_char *)&rm->rm_call.cb_cred;

  p = rc.cred + 4;			/* skip cred type */
  GETLONG(i, p); p += i;		/* skip cred data */
  if ((p - buf) + 4 > len) return;
  rc.verf = p; p += 4;			/* skip verf type */
  GETLONG(i, p); p += i;		/* skip verf data */
  if ((len -= (p - buf)) < 4) return;
  rc.args = p;
  
  switch (rc.prog) {
  case PROG_MOUNT:
    if (rc.proc != 1) return;
    if ((i = ntohl(*((u_long *)rc.args))) > 1024 || len < i + 4)
      return;
    if ((p = malloc(i + 1)) == NULL)
      nids_params.no_mem("process_rpc_client");
    strlcpy(p, rc.args + 4, i + 1);
    xid_map_enter(addr->saddr, addr->daddr, rc.xid, rc.prog, p);
    return;
    break;
  case PROG_PMAP:
    if (rc.proc != 3) return;
    if (ntohl(*((u_long *)rc.args)) == PROG_MOUNT)
      xid_map_enter(addr->saddr, addr->daddr, rc.xid, rc.prog, NULL);
    return;
    break;
  default:
    return;
    break;
  }
  /* for future handling of other RPC services */
  dr.src = addr->saddr;
  dr.dst = addr->daddr;
  dr.svc = rc.prog;
  dr.data = Buf;
  
  print_rec_uniq(&dr);
}

void
process_rpc_reply(struct tuple4 *addr, u_char *buf, int len)
{
  struct rpc_msg *rm;
  struct rpc_reply rr;
  struct xid_map_entry *xp;
  struct dsniff_rec dr;
  u_char *p;
  int i;

  rm = (struct rpc_msg *)buf;
  rr.xid = ntohl(rm->rm_xid);
  
  if ((xp = xid_map_find(addr->daddr, addr->saddr, rr.xid)) == NULL)
    return;

  if (len > 20 && ntohl(rm->rm_reply.rp_stat) == MSG_ACCEPTED) {
    rr.verf = (u_char *)&rm->rm_reply.rp_acpt.ar_verf;
    p = rr.verf + 4;				/* skip verf type */
    GETLONG(i, p); p += i;			/* skip verf data */
    if ((p - buf) + i + 4 < len) {
      GETLONG(i, p);				/* get accept_stat */
      if (i == SUCCESS) {
	dr.len = 0;
	switch (xp->prog) {
	case PROG_MOUNT:
	  dr.len = decode_rpc_mount(xp->data, p, len - (p - buf));
	  break;
	case PROG_PMAP:
	  pmap_enter(addr->saddr, ntohl(*((u_long *)p)));
	  break;
	}
	if (dr.len) {
	  dr.src = addr->daddr;
	  dr.dst = addr->saddr;
	  dr.svc = xp->prog;
	  dr.data = Buf;
	  print_rec_uniq(&dr);
	}
      }
    }
  }
  free(xp->data);
  memset(xp, 0, sizeof(*xp));
}

void
sniff_udp_client(struct ip *ip)
{
  struct udphdr *udp;
  int ip_hl = ip->ip_hl * 4;
  int ip_blen;
  
  if (ip->ip_p != IPPROTO_UDP || ntohs(ip->ip_len) - ip_hl < sizeof(*udp))
    return;
  
  ip_blen = ntohs(ip->ip_len) - ip_hl;
  udp = (struct udphdr *)((u_char *)ip + ip_hl);
  
  if (ip_blen != ntohs(udp->uh_ulen))
    return;
  
  switch (ntohs(udp->uh_dport)) {
  case 161:		/* SNMP */
    break;
  default:
    return;
    break;
  }
  process_udp_client(ip, udp);
}

void
sniff_tcp_client(struct tcp_stream *ts, void **whatever)
{
  int len;

  switch (ts->addr.dest) {
  case 21:		/* FTP */
  case 23:		/* Telnet */
  case 80:		/* HTTP */
  case 109: case 110:	/* POP */
  case 143: case 220:	/* IMAP */
  case 513: case 514:	/* R-commands */
    break;
  case 6000: case 6001: case 6002: case 6003: case 6004: case 6005: /* X11 */
    break;
  default:
    return;
    break;
  }  
  switch (ts->nids_state) {
    
  case NIDS_JUST_EST:
    ts->server.collect = 1;
    break;
    
  case NIDS_DATA:
    len = ts->server.count - ts->server.offset;

    if (len < MIN_SNAPLEN) {
      /* Not enough data yet, save buffer. */
      nids_discard (ts, 0);
    }
    else {
      /* Log and discard this stream. */
      process_tcp_client(ts, len);
      ts->server.collect = 0;
    }
    break;
    
  default:
    /* Connection closing, log what we can. */
    process_tcp_client(ts, ts->server.count - ts->server.offset);
    ts->server.collect = 0;
    break;
  }
}

void
sniff_udp_rpc(struct ip *ip)
{
  struct udphdr *udp;
  struct rpc_msg *rm;
  int ip_hl = ip->ip_hl * 4;
  int ip_blen, udp_blen;
  struct tuple4 addr;

  /* XXX - basic UDP validation. Need to implement nids_register_udp(). */
  if (ip->ip_p != IPPROTO_UDP || ntohs(ip->ip_len) - ip_hl < sizeof(*udp))
    return;
  
  ip_blen = ntohs(ip->ip_len) - ip_hl;
  udp = (struct udphdr *)((u_char *)ip + ip_hl);
  
  if (ip_blen != ntohs(udp->uh_ulen)) /* XXX - truncated? */
    return;
  
  if ((udp_blen = ip_blen - sizeof(*udp)) < 8) return;
  
  addr.saddr = ip->ip_src.s_addr;
  addr.daddr = ip->ip_dst.s_addr;
  addr.source = ntohs(udp->uh_sport);
  addr.dest = ntohs(udp->uh_dport);

  rm = (struct rpc_msg *)(udp + 1);
  
  /* Only handle RPC call/replies for registered RPC services. */
  if (pmap_find(addr.daddr, addr.dest) && ntohl
      (rm->rm_direction) == CALL) {
    process_rpc_call(&addr, (u_char *)rm, udp_blen);
  }
  else if (pmap_find(addr.saddr, addr.source) &&
	   ntohl(rm->rm_direction) == REPLY) {
    process_rpc_reply(&addr, (u_char *)rm, udp_blen);
  }
}

void
sniff_tcp_rpc(struct tcp_stream *ts, void **conn_save)
{
  struct rpc_msg *rm;
  struct rpc_record *rr;
  struct tuple4 addr;
  u_char *buf;
  int i, is_client;

  /* Only handle RPC call/replies for registered RPC services. */
  if (!pmap_find(ts->addr.daddr, ts->addr.dest))
    return;
  
  switch (ts->nids_state) {
    
  case NIDS_JUST_EST:
    ts->client.collect = 1;
    ts->server.collect = 1;
    
    if ((rr = calloc(sizeof(*rr) * 2, 1)) == NULL)
      nids_params.no_mem("sniff_tcp_rpc");
    
    *conn_save = (void *)rr;
    break;
    
  case NIDS_DATA:
    is_client = ts->server.count_new;

    if (is_client) {
      rr = (struct rpc_record *)*conn_save;
      buf = ts->server.data;
      i = ts->server.count;
      addr = ts->addr;
    }
    else {
      rr = (struct rpc_record *)*conn_save + 1;
      buf = ts->client.data;
      i = ts->client.count;
      addr.saddr = ts->addr.daddr;
      addr.daddr = ts->addr.saddr;
      addr.source = ts->addr.dest;
      addr.dest = ts->addr.source;
    }

    /* Process RPC records. */
    if (rr->rm == 0) {
      rr->rm = ntohl(*((u_long *)buf));
    }
    /* Check for complete fragment. */
    if (i < 4 + FRAGLEN(rr->rm)) {
      nids_discard(ts, 0);
      return;
    }
    /* We have a complete fragment. Build up message. */
    if (rr->data == NULL && (rr->data = malloc(FRAGLEN(rr->rm))) == NULL)
      nids_params.no_mem("sniff_tcp_rpc");
    else if ((rr->data = realloc(rr->data, rr->len + FRAGLEN(rr->rm))) == NULL)
      nids_params.no_mem("sniff_tcp_rpc");
    
    memcpy(rr->data + rr->len, buf + 4, FRAGLEN(rr->rm));
    rr->len += FRAGLEN(rr->rm);

    /* Check for complete message. */
    if (!LASTFRAG(rr->rm)) {
      rr->rm = 0;
      nids_discard(ts, 4 + FRAGLEN(rr->rm));
      return;
    }
    rr->rm = 0;

    if (rr->len < 8) return;
    rm = (struct rpc_msg *)rr->data;
    
    if (is_client && ntohl(rm->rm_direction) == CALL) {
      process_rpc_call(&addr, rr->data, rr->len);
    }
    else if (!is_client && ntohl(rm->rm_direction) == REPLY &&
	     ntohl(rm->rm_reply.rp_stat) == MSG_ACCEPTED) {
      process_rpc_reply(&addr, rr->data, rr->len);
    }
    break;
    
  default:
    rr = (struct rpc_record *)*conn_save;
    for (i = 0; i < 2; i++) {
      if (rr[i].data != NULL)
	free(rr[i].data);
    }
    free(rr);
    ts->client.collect = ts->server.collect = 0;
    break;
  }
}

void
null_syslog(int type, int errnum, struct ip *iph, void *data)
{
}

int
main(int argc, char *argv[])
{
  char c;
  char *dbname = NULL;

  while ((c = getopt(argc, argv, "i:nr:w:h?V")) != EOF) {
    switch (c) {
    case 'i':
      nids_params.device = optarg;
      break;
    case 'n':
      Opt_dns = 0;
      break;
    case 'r':
      Opt_read = 1;
      dbname = optarg;
      break;
    case 'w':
      Opt_write = 1;
      dbname = optarg;
      break;
    case 'V':
      fprintf(stderr, "Version: %s\n", VERSION);
      usage();
      break;
    default:
      usage();
    }
  }
  argc -= optind;
  argv += optind;

  if (argc != 0)
    usage();

  if (Opt_read) {
    if ((db = dbopen(dbname, O_RDONLY, 0, DB_BTREE, NULL)) == NULL) {
      perror("dbopen");
      exit(1);
    }
    print_db(db);
    exit(0);
  }
  if ((db = dbopen(dbname, O_RDWR|O_CREAT, S_IRUSR|S_IWUSR,
		   DB_BTREE, NULL)) == NULL) {
    perror("dbopen");
    exit(1);
  }
  memset(&pmap_map, 0, sizeof(pmap_map));
  memset(&xid_map, 0, sizeof(xid_map));
  
  nids_params.scan_num_hosts = 0;
  nids_params.syslog = null_syslog;

  if (!nids_init()) {
    fprintf (stderr, "%s\n", nids_errbuf);
    exit(1);
  }
  nids_register_ip(sniff_udp_client);
  nids_register_ip(sniff_udp_rpc);
  nids_register_tcp(sniff_tcp_client);
  nids_register_tcp(sniff_tcp_rpc);
  
  nids_run();

  /* NOTREACHED */
  
  exit(0);
}

/* 5000. */
