/*
 * cryptoaux.c
 *
 * Copyright (c) 2002 Dug Song <dugsong@monkey.org>
 *
 * $Id: cryptoaux.c,v 1.3 2002/11/08 21:04:52 dugsong Exp $
 */

#include <sys/types.h>
#include <openssl/ssl.h>

#include "cryptoaux.h"

DH *
DH_new_group1(void)
{
	static char *gen = "2", *group1 =
	    "FFFFFFFF" "FFFFFFFF" "C90FDAA2" "2168C234" "C4C6628B" "80DC1CD1"
	    "29024E08" "8A67CC74" "020BBEA6" "3B139B22" "514A0879" "8E3404DD"
	    "EF9519B3" "CD3A431B" "302B0A6D" "F25F1437" "4FE1356D" "6D51C245"
	    "E485B576" "625E7EC6" "F44C42E9" "A637ED6B" "0BFF5CB6" "F406B7ED"
	    "EE386BFB" "5A899FA5" "AE9F2411" "7C4B1FE6" "49286651" "ECE65381"
	    "FFFFFFFF" "FFFFFFFF";
	DH *dh;

	if ((dh = DH_new()) != NULL) {
		BN_hex2bn(&dh->p, group1);
		BN_hex2bn(&dh->g, gen);
	}
	return (dh);
}

int
BN_size(BIGNUM *bn)
{
	return (2 + BN_num_bytes(bn));
}

int
BN_put(BIGNUM *bn, u_char *buf, u_int size)
{
	if (size < (u_int)(2 + BN_num_bytes(bn)))
		return (-1);
	
	*(uint16_t *)buf = htons(BN_num_bytes(bn));
	
	return (2 + BN_bn2bin(bn, buf + 2));
}

int
BN_get(BIGNUM *bn, const u_char *buf, u_int size)
{
	uint16_t bnlen;

	if (size < 2 || (bnlen = ntohs(*(uint16_t *)buf)) > size)
		return (-1);
	
	BN_bin2bn(buf + 2, bnlen, bn);
	
	return (2 + bnlen);
}

/*
 * from OpenSSH key.c
 */
static char *
key_fingerprint_bubblebabble(u_char *dgst_raw, u_int dgst_raw_len)
{
	char vowels[] = { 'a', 'e', 'i', 'o', 'u', 'y' };
	char consonants[] = { 'b', 'c', 'd', 'f', 'g', 'h', 'k', 'l', 'm',
			      'n', 'p', 'r', 's', 't', 'v', 'z', 'x' };
	u_int i, j = 0, rounds, seed = 1;
	char *retval;
	
	rounds = (dgst_raw_len / 2) + 1;
	retval = malloc(sizeof(char) * (rounds*6));
	retval[j++] = 'x';
	for (i = 0; i < rounds; i++) {
		u_int idx0, idx1, idx2, idx3, idx4;
		if ((i + 1 < rounds) || (dgst_raw_len % 2 != 0)) {
			idx0 = (((((u_int)(dgst_raw[2 * i])) >> 6) & 3) +
			    seed) % 6;
			idx1 = (((u_int)(dgst_raw[2 * i])) >> 2) & 15;
			idx2 = ((((u_int)(dgst_raw[2 * i])) & 3) +
			    (seed / 6)) % 6;
			retval[j++] = vowels[idx0];
			retval[j++] = consonants[idx1];
			retval[j++] = vowels[idx2];
			if ((i + 1) < rounds) {
				idx3 = (((u_int)(dgst_raw[(2 * i) + 1])) >> 4) & 15;
				idx4 = (((u_int)(dgst_raw[(2 * i) + 1]))) & 15;
				retval[j++] = consonants[idx3];
				retval[j++] = '-';
				retval[j++] = consonants[idx4];
				seed = ((seed * 5) +
				    ((((u_int)(dgst_raw[2 * i])) * 7) +
				    ((u_int)(dgst_raw[(2 * i) + 1])))) % 36;
			}
		} else {
			idx0 = seed % 6;
			idx1 = 16;
			idx2 = seed / 6;
			retval[j++] = vowels[idx0];
			retval[j++] = consonants[idx1];
			retval[j++] = vowels[idx2];
		}
	}
	retval[j++] = 'x';
	retval[j++] = '\0';

	return (retval);
}

char *
EVP_PKEY_fingerprint(EVP_PKEY *pkey)
{
	EVP_MD_CTX md;
	u_char *buf, digest[EVP_MAX_MD_SIZE];
	int nlen, elen, dlen;
	
	if (pkey->type != EVP_PKEY_RSA)
		return (NULL);
	
	nlen = BN_num_bytes(pkey->pkey.rsa->n);
	elen = BN_num_bytes(pkey->pkey.rsa->e);
	
	if ((buf = malloc(nlen + elen)) == NULL)
		return (NULL);

	BN_bn2bin(pkey->pkey.rsa->n, buf);
	BN_bn2bin(pkey->pkey.rsa->e, buf + nlen);
	
	EVP_DigestInit(&md, EVP_sha1());
	EVP_DigestUpdate(&md, buf, nlen + elen);
	EVP_DigestFinal(&md, digest, &dlen);

	free(buf);
	
	return (key_fingerprint_bubblebabble(digest, dlen));
}
