/*
 * ctx.c
 *
 * Copyright (c) 2002 Dug Song <dugsong@monkey.org>
 *
 * $Id: ctx.c,v 1.1 2002/11/08 21:04:52 dugsong Exp $
 */

#include "w00gaim.h"

static GHashTable	*w00ctx_tab;

void
w00gaim_ctx_init(void)
{
	w00ctx_tab = g_hash_table_new(g_str_hash, g_str_equal);
}

struct w00ctx *
w00gaim_ctx_new(const char *my_name, const char *her_name)
{
	struct w00ctx *ctx = g_new0(struct w00ctx, 1);

	ctx->my_pkey = w00gaim_pkey_get(my_name, TRUE);
	ctx->her_pkey = w00gaim_pkey_get(her_name, FALSE);
	
	if (ctx->my_pkey == NULL || ctx->her_pkey == NULL)
		return (w00gaim_ctx_free(ctx));
	
	ctx->her_name = g_strdup(her_name);
	g_hash_table_insert(w00ctx_tab, ctx->her_name, ctx);
	
	return (ctx);
}

struct w00ctx *
w00gaim_ctx_lookup(const char *who)
{
	return (g_hash_table_lookup(w00ctx_tab, who));
}

void
w00gaim_ctx_remove(struct w00ctx *ctx)
{
	g_hash_table_remove(w00ctx_tab, ctx->her_name);
}

int
w00gaim_ctx_encrypt(struct w00ctx *ctx, char **outbuf, int *outlen,
    char *inbuf, int inlen)
{
	u_int blocklen, enclen, padlen;

	/* Calculate message length. */
	blocklen = EVP_CIPHER_CTX_block_size(&ctx->enc_ctx);
	RAND_pseudo_bytes((char *)&padlen, sizeof(padlen));
	padlen %= 128;
	enclen = 4 + inlen + blocklen + padlen;
	enclen += blocklen - (enclen % blocklen);
	
	/* Build message. */
	*outbuf = g_malloc(4 + enclen);
	*(uint32_t *)(*outbuf) = htonl(w00MSG);
	*outlen = 4;
	
	*(uint32_t *)(*outbuf + 4) = htonl(inlen);
	memcpy(*outbuf + 8, inbuf, inlen);
	
	RAND_pseudo_bytes(*outbuf + (4 + 4 + inlen), enclen - (4 + inlen));
	
	/* Encrypt message. */
	EVP_EncryptUpdate(&ctx->enc_ctx, *outbuf + 4, &padlen,
	    *outbuf + 4, enclen);
	*outlen += padlen;
	
	return (0);
}

int
w00gaim_ctx_decrypt(struct w00ctx *ctx, char **outbuf, int *outlen,
    char *inbuf, int inlen)
{
	uint32_t len;
	
	if (inlen < 4 || ntohl(*(uint32_t *)inbuf) != w00MSG)
		return (-1);
	
	*outbuf = g_malloc(inlen + EVP_CIPHER_CTX_block_size(&ctx->dec_ctx));

	EVP_DecryptUpdate(&ctx->dec_ctx, *outbuf, outlen,
	    inbuf + 4, inlen - 4);
	
	if (4 + (len = ntohl(*(uint32_t *)(*outbuf))) > (uint32_t)*outlen) {
		g_free(*outbuf);
		return (-1);
	}
	memmove(*outbuf, *outbuf + 4, len);
	(*outbuf)[len] = '\0';
	*outlen = len;
	
	return (0);
}

int
w00gaim_ctx_kexsend(struct w00ctx *ctx, char **buf, int *len)
{
	EVP_MD_CTX md;
	int i;
	
	if (ctx->my_dh != NULL)
		DH_free(ctx->my_dh);
	
	/* Generate and send authenticated DH public value. */
	ctx->my_dh = DH_new_group1();
	DH_generate_key(ctx->my_dh);

	*len = 8 + BN_size(ctx->my_dh->pub_key) +
	    4 + EVP_PKEY_size(ctx->my_pkey) + 24;
	*buf = g_malloc(*len);

	*(uint32_t *)(*buf) = htonl(w00KEX);
	*(uint32_t *)(*buf + 4) = htonl(EVP_CIPHER_nid(ctx->cipher));
	*len = 8 + BN_put(ctx->my_dh->pub_key, *buf + 8, *len - 8);

	EVP_SignInit(&md, EVP_sha1());
	EVP_SignUpdate(&md, *buf, *len);
	EVP_SignFinal(&md, *buf + *len + 4, &i, ctx->my_pkey);
	*(uint32_t *)(*buf + *len) = htonl(i);
	*len += 4 + i;
	
	return (0);
}

int
w00gaim_ctx_kexrecv(struct w00ctx *ctx, char *buf, int len)
{
	EVP_MD_CTX md;
	char *p = buf;
	uint32_t i, n;

	if (len < 8 || ntohl(*(uint32_t *)buf) != w00KEX || (ctx->cipher =
	    EVP_get_cipherbynid(ntohl(*(uint32_t *)(buf + 4)))) == NULL)
		return (-1);
	
	buf += 8, len -= 8;
	ctx->her_bn = BN_new();

	if ((i = BN_get(ctx->her_bn, buf, len)) < 0)
		goto kexrecv_err;
	
	buf += i, len -= i;
	if (len < 4 || (n = ntohl(*(uint32_t *)buf)) > len)
		goto kexrecv_err;
	
	EVP_VerifyInit(&md, EVP_sha1());
	EVP_VerifyUpdate(&md, p, buf - p);
	if (EVP_VerifyFinal(&md, buf + 4, n, ctx->her_pkey) == 1)
		return (0);
	
 kexrecv_err:
	BN_free(ctx->her_bn);
	ctx->her_bn = NULL;
	ctx->cipher = NULL;
	
	return (-1);
}

int
w00gaim_ctx_kexsave(struct w00ctx *ctx)
{
	volatile u_char key[512];

	memset((u_char *)key, 0, sizeof(key));
	DH_compute_key((u_char *)key, ctx->her_bn, ctx->my_dh);
	EVP_EncryptInit(&ctx->enc_ctx, ctx->cipher,
	    (u_char *)key, (u_char *)key);
	EVP_DecryptInit(&ctx->dec_ctx, ctx->cipher,
	    (u_char *)key, (u_char *)key);
	EVP_CIPHER_CTX_set_padding(&ctx->enc_ctx, 0);
	EVP_CIPHER_CTX_set_padding(&ctx->dec_ctx, 0);
	memset((u_char *)key, 0, sizeof(key));
	
	return (0);
}

struct w00ctx *
w00gaim_ctx_free(struct w00ctx *ctx)
{
	if (ctx->her_name != NULL) {
		g_hash_table_remove(w00ctx_tab, ctx->her_name);
		g_free(ctx->her_name);
	}
	if (ctx->my_dh != NULL)
		DH_free(ctx->my_dh);
	if (ctx->her_bn != NULL)
		BN_free(ctx->her_bn);
	if (ctx->my_pkey != NULL)
		EVP_PKEY_free(ctx->my_pkey);
	if (ctx->her_pkey != NULL)
		EVP_PKEY_free(ctx->her_pkey);
	
	EVP_CIPHER_CTX_cleanup(&ctx->enc_ctx);
	EVP_CIPHER_CTX_cleanup(&ctx->dec_ctx);

	g_free(ctx);
	
	return (NULL);
}

static gboolean
_free_w00ctx(gpointer key, gpointer value, gpointer arg)
{
	struct w00ctx *ctx = (struct w00ctx *)value;

	/* XXX */
	g_free(ctx->her_name);
	ctx->her_name = NULL;
	w00gaim_ctx_free(ctx);
	
	return (TRUE);
}

void
w00gaim_ctx_destroy(void)
{
	g_hash_table_foreach_remove(w00ctx_tab, _free_w00ctx, NULL);
	g_hash_table_destroy(w00ctx_tab);
}
