/*
  macof.c

  C port of macof-1.1 from the Perl Net::RawIP distribution.
  Tests network devices by flooding local network with MAC-addresses.
  
  Perl macof originally written by Ian Vitek <ian.vitek@infosec.se>.
  
  Copyright (c) 1999 Dug Song <dugsong@monkey.org>
  All rights reserved, all wrongs reversed.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
  3. The name of author may not be used to endorse or promote products
     derived from this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
  AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL
  THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
  ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  $Id: macof.c,v 1.6 2000/01/04 20:25:39 dugsong Exp $
*/

#include <sys/types.h>
#include <sys/param.h>
#include <stdio.h>
#include <string.h>
#include <libnet.h>
#include <pcap.h>

#include "version.h"

extern char *ether_ntoa(struct ether_addr *);
extern struct ether_addr *ether_aton(char *);

int	Verbose = 0;
u_long	Src = 0;
u_long	Dst = 0;
u_char *Tha = NULL;
u_short	Dport = 0;
u_short Sport = 0;
char   *Intf = NULL;
int	Repeat = -1;

void
usage(void)
{
  fprintf(stderr, "Usage: macof [-v] [-s src] [-d dst] [-e tha] "
	  "[-x sport] [-y dport] [-i interface] [-n times]\n");
  exit(1);
}

void
gen_mac(u_char *mac)
{
  *((u_long *)mac) = libnet_get_prand(PRu32);
  *((u_short *)(mac + 4)) = libnet_get_prand(PRu16);
}

int
main(int argc, char *argv[])
{
  char c;
  int i;
  struct libnet_link_int *llif;
  char ebuf[PCAP_ERRBUF_SIZE];
  u_char sha[ETHER_ADDR_LEN], tha[ETHER_ADDR_LEN];
  u_long src, dst;
  u_short sport, dport;
  u_char pkt[ETH_H + IP_H + TCP_H];
  
  while ((c = getopt(argc, argv, "vs:d:e:x:y:i:n:h?V")) != EOF) {
    switch (c) {
    case 'v':
      Verbose = 1;
      break;
    case 's':
      Src = libnet_name_resolve(optarg, 0);
      break;
    case 'd':
      Dst = libnet_name_resolve(optarg, 0);
      break;
    case 'e':
      Tha = (u_char *)ether_aton(optarg);
      break;
    case 'x':
      Sport = atoi(optarg);
      break;
    case 'y':
      Dport = atoi(optarg);
      break;
    case 'i':
      Intf = optarg;
      break;
    case 'n':
      Repeat = atoi(optarg);
      break;
    case 'V':
      fprintf(stderr, "Version: %s\n", VERSION);
      usage();
      break;
    default:
      usage();
      break;
    }
  }
  argc -= optind;
  argv += optind;

  if (argc != 0)
    usage();

  if (!Intf && (Intf = pcap_lookupdev(ebuf)) == NULL) {
    fprintf(stderr, "%s\n", ebuf);
    exit(1);
  }
  if ((llif = libnet_open_link_interface(Intf, ebuf)) == 0) {
    fprintf(stderr, "%s\n", ebuf);
    exit(1);
  }
  libnet_seed_prand();
  
  for (i = 0; i != Repeat; i++) {
    gen_mac(sha);
    if (Verbose)
      printf("%s", ether_ntoa((struct ether_addr *)sha));
    
    if (Tha != NULL) {
      memcpy(tha, Tha, sizeof(tha));
      if (Verbose)
	printf("\n");
    }
    else {
      gen_mac(tha);
      if (Verbose)
	printf("\t%s\n", ether_ntoa((struct ether_addr *)tha));
    }
    
    if (Src != 0) src = Src;
    else src = libnet_get_prand(PRu32);

    if (Dst != 0) dst = Dst;
    else dst = libnet_get_prand(PRu32);

    if (Sport != 0) sport = Sport;
    else sport = libnet_get_prand(PRu16);

    if (Dport != 0) dport = Dport;
    else dport = libnet_get_prand(PRu16);

    libnet_build_ethernet(tha, sha, ETHERTYPE_IP, NULL, 0, pkt);

    libnet_build_ip(TCP_H, 0, libnet_get_prand(PRu16), 0, 64, IPPROTO_TCP,
		    src, dst, NULL, 0, pkt + ETH_H);

    libnet_build_tcp(sport, dport, libnet_get_prand(PRu32),
		     libnet_get_prand(PRu32), TH_SYN, 1024, 0, NULL, 0,
		     pkt + ETH_H + IP_H);

    libnet_do_checksum(pkt + ETH_H, IPPROTO_IP, IP_H);
    libnet_do_checksum(pkt + ETH_H, IPPROTO_TCP, TCP_H);

    if (libnet_write_link_layer(llif, Intf, pkt, sizeof(pkt)) != sizeof(pkt)) {
      perror("write");
      exit(1);
    }
  }
  exit(0);
}

/* 5000 */
